<?php
/**
 * Simple OTP Manager for Razorhost
 * Uses PHP mail() function - works everywhere
 * No SMTP complications
 */

class SimpleOTPManager {
    private $db;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    /**
     * Generate and send OTP to email - SIMPLE VERSION
     */
    public function generateAndSendOTP($email, $name = '') {
        try {
            // Validate email
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                return ['success' => false, 'message' => 'Invalid email address'];
            }
            
            // Generate 6-digit OTP
            $otp = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
            
            // Calculate expiry time (5 minutes from now)
            $expiryTime = date('Y-m-d H:i:s', time() + (5 * 60));
            $createdAt = date('Y-m-d H:i:s');
            
            // Delete any old OTPs for this email
            $this->db->query('DELETE FROM email_otps WHERE email = ?', [$email]);
            
            // Store OTP in database
            $this->db->insert('email_otps', [
                'email' => $email,
                'otp' => $otp,
                'attempts' => 0,
                'created_at' => $createdAt,
                'expires_at' => $expiryTime
            ]);
            
            // Send email
            $sendResult = $this->sendOTPEmailSimple($email, $otp, $name);
            
            if ($sendResult) {
                return ['success' => true, 'message' => 'OTP sent to ' . $email];
            } else {
                // OTP stored but email failed
                error_log("OTP Email Failed for: $email");
                return ['success' => false, 'message' => 'Failed to send email. Contact admin.'];
            }
            
        } catch (Exception $e) {
            error_log('OTP Generation Error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error: ' . $e->getMessage()];
        }
    }
    
    /**
     * Send OTP via simple PHP mail()
     */
    private function sendOTPEmailSimple($email, $otp, $name = '') {
        try {
            $subject = 'Your OTP Code - ' . APP_NAME;
            
            // Simple but effective email body
            $emailBody = "
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; background: #f5f5f5; }
                    .container { background: white; max-width: 500px; margin: 30px auto; padding: 40px; border-radius: 10px; }
                    .header { text-align: center; margin-bottom: 30px; color: #667eea; font-size: 20px; font-weight: bold; }
                    .content { text-align: center; }
                    .otp-box { background: #667eea; color: white; padding: 20px; border-radius: 8px; margin: 20px 0; font-size: 32px; font-weight: bold; letter-spacing: 5px; }
                    .footer { text-align: center; color: #999; font-size: 12px; margin-top: 30px; }
                </style>
            </head>
            <body>
                <div class='container'>
                    <div class='header'>🔐 " . APP_NAME . "</div>
                    <div class='content'>
                        <h2>Email Verification</h2>
                        <p>Hi " . htmlspecialchars($name) . ",</p>
                        <p>Your OTP verification code is:</p>
                        <div class='otp-box'>$otp</div>
                        <p style='color: #666;'>This code will expire in 5 minutes</p>
                        <p style='color: #666;'>If you didn't request this, please ignore this email</p>
                    </div>
                    <div class='footer'>
                        <p>&copy; " . date('Y') . " " . APP_NAME . ". All rights reserved.</p>
                    </div>
                </div>
            </body>
            </html>
            ";
            
            // Prepare headers
            $headers = "MIME-Version: 1.0" . "\r\n";
            $headers .= "Content-type: text/html; charset=UTF-8" . "\r\n";
            $headers .= "From: " . APP_NAME . " <noreply@simplechecks.xyz>" . "\r\n";
            $headers .= "Reply-To: support@simplechecks.xyz" . "\r\n";
            
            // Send email using PHP mail()
            $result = @mail($email, $subject, $emailBody, $headers);
            
            if (!$result) {
                error_log("PHP mail() failed for: $email");
                return false;
            }
            
            return true;
            
        } catch (Exception $e) {
            error_log('Email Send Error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Verify OTP
     */
    public function verifyOTP($email, $otp) {
        try {
            $otp = preg_replace('/\D/', '', $otp); // Remove non-digits
            
            // Get OTP record
            $otpRecord = $this->db->fetch(
                'SELECT id, otp, attempts, expires_at FROM email_otps WHERE email = ? ORDER BY created_at DESC LIMIT 1',
                [$email]
            );
            
            if (!$otpRecord) {
                return ['success' => false, 'message' => 'No OTP found. Please register again.'];
            }
            
            // Check expiry
            if (strtotime($otpRecord['expires_at']) < time()) {
                $this->db->query('DELETE FROM email_otps WHERE id = ?', [$otpRecord['id']]);
                return ['success' => false, 'message' => 'OTP expired. Please register again.'];
            }
            
            // Check attempts
            if ($otpRecord['attempts'] >= 5) {
                return ['success' => false, 'message' => 'Max attempts exceeded. Please register again.'];
            }
            
            // Verify OTP
            if ($otpRecord['otp'] !== $otp) {
                // Increment attempts
                $this->db->update(
                    'email_otps',
                    ['attempts' => $otpRecord['attempts'] + 1],
                    'id = ?',
                    [$otpRecord['id']]
                );
                
                $remaining = 5 - ($otpRecord['attempts'] + 1);
                return ['success' => false, 'message' => "Invalid OTP. $remaining attempts left."];
            }
            
            // OTP correct - delete it
            $this->db->query('DELETE FROM email_otps WHERE id = ?', [$otpRecord['id']]);
            
            return ['success' => true, 'message' => 'OTP verified successfully'];
            
        } catch (Exception $e) {
            error_log('OTP Verification Error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Verification failed'];
        }
    }
    
    /**
     * Get remaining time for OTP
     */
    public function getRemainingTime($email) {
        try {
            $otpRecord = $this->db->fetch(
                'SELECT expires_at FROM email_otps WHERE email = ? ORDER BY created_at DESC LIMIT 1',
                [$email]
            );
            
            if (!$otpRecord) {
                return 0;
            }
            
            $remaining = strtotime($otpRecord['expires_at']) - time();
            return max(0, $remaining);
            
        } catch (Exception $e) {
            error_log('Error getting remaining time: ' . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Resend OTP
     */
    public function resendOTP($email) {
        try {
            // Get existing OTP
            $otpRecord = $this->db->fetch(
                'SELECT otp, expires_at FROM email_otps WHERE email = ? ORDER BY created_at DESC LIMIT 1',
                [$email]
            );
            
            if (!$otpRecord) {
                return ['success' => false, 'message' => 'No OTP found'];
            }
            
            // Send existing OTP again
            $sendResult = $this->sendOTPEmailSimple($email, $otpRecord['otp']);
            
            if ($sendResult) {
                return ['success' => true, 'message' => 'OTP resent to ' . $email];
            } else {
                return ['success' => false, 'message' => 'Failed to resend OTP'];
            }
            
        } catch (Exception $e) {
            error_log('Resend OTP Error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error: ' . $e->getMessage()];
        }
    }
}
?>
