<?php
// classes/SubscriptionManager.php

class SubscriptionManager {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    // ========== FREE TRIAL MANAGEMENT ==========
    
    /**
     * Initialize free trial for new user (10 searches)
     */
    public function initializeFreeTrial($userId) {
        try {
            $data = [
                'user_id' => $userId,
                'trial_searches_limit' => 10,
                'trial_searches_used' => 0,
                'is_trial_active' => true,
                'trial_started_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('free_trial_tracking', $data);
            return ['status' => true, 'message' => 'Free trial initialized'];
        } catch (Exception $e) {
            error_log('Initialize Free Trial Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Failed to initialize trial'];
        }
    }
    
    /**
     * Get user's free trial info
     */
    public function getFreeTrial($userId) {
        $result = $this->db->fetch(
            'SELECT * FROM free_trial_tracking WHERE user_id = ?',
            [$userId]
        );
        
        if (!$result) {
            return null;
        }
        
        return [
            'user_id' => $result['user_id'],
            'searches_used' => (int)$result['trial_searches_used'],
            'searches_remaining' => (int)$result['trial_searches_limit'] - (int)$result['trial_searches_used'],
            'is_trial_active' => (bool)$result['is_trial_active'],
            'is_active' => (bool)$result['is_trial_active'],
            'started_at' => $result['trial_started_at']
        ];
    }
    
    /**
     * Deduct from free trial
     */
    public function deductFreeTrialSearch($userId) {
        try {
            $trial = $this->getFreeTrial($userId);
            
            if (!$trial || !$trial['is_active'] || $trial['searches_remaining'] <= 0) {
                return ['status' => false, 'message' => 'Free trial not available'];
            }
            
            $this->db->query(
                'UPDATE free_trial_tracking SET trial_searches_used = trial_searches_used + 1 WHERE user_id = ?',
                [$userId]
            );
            
            $updatedTrial = $this->getFreeTrial($userId);
            
            // If trial exhausted, mark it as inactive
            if ($updatedTrial['searches_remaining'] <= 0) {
                $this->db->query(
                    'UPDATE free_trial_tracking SET is_trial_active = false, trial_ended_at = NOW() WHERE user_id = ?',
                    [$userId]
                );
            }
            
            return ['status' => true, 'trial' => $updatedTrial];
        } catch (Exception $e) {
            error_log('Deduct Free Trial Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Error processing trial'];
        }
    }
    
    // ========== SUBSCRIPTION PLANS MANAGEMENT ==========
    
    /**
     * Get all active subscription plans
     */
    public function getActivePlans() {
        $result = $this->db->fetchAll(
            'SELECT * FROM subscription_plans WHERE is_active = true ORDER BY duration_days ASC'
        );
        return $result ?? [];
    }
    
    /**
     * Get plan by ID
     */
    public function getPlanById($planId) {
        return $this->db->fetch(
            'SELECT * FROM subscription_plans WHERE id = ? AND is_active = true',
            [$planId]
        );
    }
    
    /**
     * Create new subscription plan (admin only)
     */
    public function createPlan($planName, $durationDays, $searchLimit, $price, $description = '') {
        try {
            $data = [
                'plan_name' => $planName,
                'duration_days' => $durationDays,
                'search_limit' => $searchLimit,
                'price' => $price,
                'description' => $description,
                'is_active' => true
            ];
            
            $this->db->insert('subscription_plans', $data);
            return ['status' => true, 'message' => 'Plan created successfully'];
        } catch (Exception $e) {
            error_log('Create Plan Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Failed to create plan'];
        }
    }
    
    /**
     * Update subscription plan (admin only)
     */
    public function updatePlan($planId, $planName, $durationDays, $searchLimit, $price, $description = '', $isActive = true) {
        try {
            $this->db->query(
                'UPDATE subscription_plans SET plan_name = ?, duration_days = ?, search_limit = ?, price = ?, description = ?, is_active = ? WHERE id = ?',
                [$planName, $durationDays, $searchLimit, $price, $description, $isActive ? 1 : 0, $planId]
            );
            return ['status' => true, 'message' => 'Plan updated successfully'];
        } catch (Exception $e) {
            error_log('Update Plan Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Failed to update plan'];
        }
    }
    
    /**
     * Delete subscription plan (admin only)
     */
    public function deletePlan($planId) {
        try {
            $this->db->query(
                'UPDATE subscription_plans SET is_active = false WHERE id = ?',
                [$planId]
            );
            return ['status' => true, 'message' => 'Plan deleted successfully'];
        } catch (Exception $e) {
            error_log('Delete Plan Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Failed to delete plan'];
        }
    }
    
    // ========== USER SUBSCRIPTION MANAGEMENT ==========
    
    /**
     * Get active subscription for user
     */
    public function getActiveSubscription($userId) {
        $result = $this->db->fetch(
            'SELECT us.*, sp.plan_name, sp.search_limit, sp.duration_days 
             FROM user_subscriptions us
             LEFT JOIN subscription_plans sp ON us.plan_id = sp.id
             WHERE us.user_id = ? AND us.status = "active" AND us.end_date > NOW()
             ORDER BY us.created_at DESC LIMIT 1',
            [$userId]
        );
        
        if (!$result) {
            return null;
        }
        
        return [
            'id' => $result['id'],
            'plan_id' => $result['plan_id'],
            'plan_name' => $result['plan_name'],
            'searches_used' => (int)$result['search_used'],
            'searches_limit' => (int)$result['search_limit'],
            'searches_remaining' => (int)$result['search_limit'] - (int)$result['search_used'],
            'start_date' => $result['start_date'],
            'end_date' => $result['end_date'],
            'days_remaining' => $this->getDaysRemaining($result['end_date']),
            'status' => $result['status'],
            'razorpay_payment_id' => $result['razorpay_payment_id']
        ];
    }
    
    /**
     * Create new subscription from plan
     */
    public function createSubscription($userId, $planId, $razorpayOrderId) {
        try {
            $plan = $this->getPlanById($planId);
            
            if (!$plan) {
                return ['status' => false, 'message' => 'Plan not found'];
            }
            
            $startDate = date('Y-m-d H:i:s');
            $endDate = date('Y-m-d H:i:s', strtotime('+' . $plan['duration_days'] . ' days'));
            
            $data = [
                'user_id' => $userId,
                'plan_id' => $planId,
                'razorpay_order_id' => $razorpayOrderId,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'status' => 'active',
                'payment_status' => 'pending'
            ];
            
            $subscriptionId = $this->db->insert('user_subscriptions', $data);
            
            // Log to history
            $this->logSubscriptionHistory($userId, $subscriptionId, 'created', null, 'active', "Subscription created for plan {$plan['plan_name']}");
            
            return [
                'status' => true,
                'subscription_id' => $subscriptionId,
                'message' => 'Subscription created successfully'
            ];
        } catch (Exception $e) {
            error_log('Create Subscription Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Failed to create subscription'];
        }
    }
    
    /**
     * Confirm payment and activate subscription
     */
    public function confirmPayment($userId, $subscriptionId, $razorpayPaymentId, $razorpaySignature) {
        try {
            $this->db->query(
                'UPDATE user_subscriptions 
                 SET razorpay_payment_id = ?, razorpay_signature = ?, payment_status = "success", status = "active"
                 WHERE id = ? AND user_id = ?',
                [$razorpayPaymentId, $razorpaySignature, $subscriptionId, $userId]
            );
            
            $this->logSubscriptionHistory($userId, $subscriptionId, 'payment_confirmed', 'pending', 'active', "Payment confirmed: {$razorpayPaymentId}");
            
            return ['status' => true, 'message' => 'Payment confirmed successfully'];
        } catch (Exception $e) {
            error_log('Confirm Payment Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Failed to confirm payment'];
        }
    }
    
    /**
     * Deduct search from active subscription
     */
    public function deductSubscriptionSearch($userId) {
        try {
            $subscription = $this->getActiveSubscription($userId);
            
            if (!$subscription || $subscription['searches_remaining'] <= 0) {
                return ['status' => false, 'message' => 'No active subscription or search limit reached'];
            }
            
            $this->db->query(
                'UPDATE user_subscriptions SET search_used = search_used + 1 WHERE id = ?',
                [$subscription['id']]
            );
            
            $updatedSubscription = $this->getActiveSubscription($userId);
            
            return ['status' => true, 'subscription' => $updatedSubscription];
        } catch (Exception $e) {
            error_log('Deduct Subscription Search Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Error processing search'];
        }
    }
    
    /**
     * Get all subscriptions for user (including expired)
     */
    public function getUserSubscriptions($userId) {
        $result = $this->db->fetchAll(
            'SELECT us.*, sp.plan_name, sp.search_limit, sp.price 
             FROM user_subscriptions us
             JOIN subscription_plans sp ON us.plan_id = sp.id
             WHERE us.user_id = ?
             ORDER BY us.created_at DESC',
            [$userId]
        );
        
        return $result ?? [];
    }
    
    /**
     * Cancel active subscription
     */
    public function cancelSubscription($userId, $subscriptionId) {
        try {
            $this->db->query(
                'UPDATE user_subscriptions SET status = "cancelled" WHERE id = ? AND user_id = ?',
                [$subscriptionId, $userId]
            );
            
            $this->logSubscriptionHistory($userId, $subscriptionId, 'cancelled', 'active', 'cancelled', 'Subscription cancelled by user');
            
            return ['status' => true, 'message' => 'Subscription cancelled successfully'];
        } catch (Exception $e) {
            error_log('Cancel Subscription Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Failed to cancel subscription'];
        }
    }
    
    // ========== UTILITY METHODS ==========
    
    /**
     * Calculate days remaining
     */
    private function getDaysRemaining($endDate) {
        $now = new DateTime();
        $end = new DateTime($endDate);
        $interval = $now->diff($end);
        return max(0, $interval->days);
    }
    
    /**
     * Log subscription history
     */
    private function logSubscriptionHistory($userId, $subscriptionId, $action, $oldStatus, $newStatus, $details = '') {
        try {
            $data = [
                'user_id' => $userId,
                'subscription_id' => $subscriptionId,
                'action' => $action,
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
                'details' => json_encode(['message' => $details])
            ];
            
            $this->db->insert('subscription_history', $data);
        } catch (Exception $e) {
            error_log('Log Subscription History Error: ' . $e->getMessage());
        }
    }
    
    /**
     * Check if user has access to search
     */
    public function canUserSearch($userId) {
        // Check free trial first
        $trial = $this->getFreeTrial($userId);
        if ($trial && $trial['is_active'] && $trial['searches_remaining'] > 0) {
            return ['status' => true, 'type' => 'trial'];
        }
        
        // Check active subscription
        $subscription = $this->getActiveSubscription($userId);
        if ($subscription && $subscription['searches_remaining'] > 0) {
            return ['status' => true, 'type' => 'subscription'];
        }
        
        return ['status' => false, 'message' => 'No active trial or subscription'];
    }
    
    /**
     * Process search and deduct from appropriate source
     */
    public function processSearch($userId) {
        // First check if trial is active
        $trial = $this->getFreeTrial($userId);
        if ($trial && $trial['is_active'] && $trial['searches_remaining'] > 0) {
            return $this->deductFreeTrialSearch($userId);
        }
        
        // Then check subscription
        $subscription = $this->getActiveSubscription($userId);
        if ($subscription && $subscription['searches_remaining'] > 0) {
            return $this->deductSubscriptionSearch($userId);
        }
        
        return ['status' => false, 'message' => 'No available searches'];
    }
    
    /**
     * Get subscription stats for admin
     */
    public function getSubscriptionStats() {
        $stats = [
            'total_plans' => $this->db->fetch('SELECT COUNT(*) as count FROM subscription_plans WHERE is_active = true')['count'],
            'total_subscriptions' => $this->db->fetch('SELECT COUNT(*) as count FROM user_subscriptions')['count'],
            'active_subscriptions' => $this->db->fetch('SELECT COUNT(*) as count FROM user_subscriptions WHERE status = "active" AND end_date > NOW()')['count'],
            'revenue' => $this->db->fetch('SELECT SUM(sp.price) as total FROM user_subscriptions us JOIN subscription_plans sp ON us.plan_id = sp.id WHERE us.payment_status = "success"')['total'] ?? 0,
            'users_on_trial' => $this->db->fetch('SELECT COUNT(*) as count FROM free_trial_tracking WHERE is_trial_active = true')['count']
        ];
        
        return $stats;
    }
}
?>
