<?php
/**
 * Verify and Add Login Tracking Columns
 * Visit: http://localhost/extract/database/verify_login_columns.php
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../classes/Database.php';

$db = Database::getInstance();

echo "<style>
body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
.container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
.success { background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; color: #155724; margin: 10px 0; }
.error { background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; color: #721c24; margin: 10px 0; }
.info { background: #d1ecf1; border: 1px solid #bee5eb; padding: 15px; border-radius: 5px; color: #0c5460; margin: 10px 0; }
table { width: 100%; border-collapse: collapse; margin: 10px 0; }
table td, table th { padding: 10px; border: 1px solid #ddd; text-align: left; }
table th { background: #f2f2f2; font-weight: bold; }
code { background: #f4f4f4; padding: 2px 5px; border-radius: 3px; }
</style>";

echo "<div class='container'>";
echo "<h2>🔍 Login Tracking Columns Verification</h2>";

try {
    // Check current columns
    $columns = $db->query("SHOW COLUMNS FROM users")->fetchAll(PDO::FETCH_ASSOC);
    $columnNames = array_column($columns, 'Field');
    
    echo "<h3>Current Users Table Columns:</h3>";
    echo "<table>";
    echo "<tr><th>Column Name</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
    foreach ($columns as $col) {
        echo "<tr>";
        echo "<td>" . $col['Field'] . "</td>";
        echo "<td>" . $col['Type'] . "</td>";
        echo "<td>" . $col['Null'] . "</td>";
        echo "<td>" . ($col['Key'] ?: '-') . "</td>";
        echo "<td>" . ($col['Default'] ?: 'NULL') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    echo "<h3>Required Columns Status:</h3>";
    
    $required = ['login_device', 'login_ip', 'last_login_at'];
    $missing = [];
    
    foreach ($required as $colName) {
        if (in_array($colName, $columnNames)) {
            echo "<div class='success'>✓ Column '<code>$colName</code>' EXISTS</div>";
        } else {
            echo "<div class='error'>✗ Column '<code>$colName</code>' MISSING</div>";
            $missing[] = $colName;
        }
    }
    
    // If columns are missing, add them
    if (!empty($missing)) {
        echo "<h3>Adding Missing Columns...</h3>";
        
        if (in_array('login_device', $missing)) {
            $db->query("ALTER TABLE users ADD COLUMN login_device VARCHAR(50) DEFAULT NULL");
            echo "<div class='success'>✓ Added login_device column</div>";
        }
        
        if (in_array('login_ip', $missing)) {
            $db->query("ALTER TABLE users ADD COLUMN login_ip VARCHAR(45) DEFAULT NULL");
            echo "<div class='success'>✓ Added login_ip column</div>";
        }
        
        if (in_array('last_login_at', $missing)) {
            $db->query("ALTER TABLE users ADD COLUMN last_login_at DATETIME DEFAULT NULL");
            echo "<div class='success'>✓ Added last_login_at column</div>";
        }
        
        echo "<div class='info'><strong>All missing columns have been added!</strong></div>";
    } else {
        echo "<div class='success'><strong>✓ All required columns are present!</strong></div>";
    }
    
    // Show sample data
    echo "<h3>Sample User Data:</h3>";
    $users = $db->query("SELECT id, name, email, login_device, login_ip, last_login_at FROM users LIMIT 5")->fetchAll(PDO::FETCH_ASSOC);
    
    if (!empty($users)) {
        echo "<table>";
        echo "<tr><th>ID</th><th>Name</th><th>Email</th><th>Device</th><th>IP</th><th>Last Login</th></tr>";
        foreach ($users as $user) {
            echo "<tr>";
            echo "<td>" . $user['id'] . "</td>";
            echo "<td>" . htmlspecialchars($user['name']) . "</td>";
            echo "<td>" . htmlspecialchars($user['email']) . "</td>";
            echo "<td>" . ($user['login_device'] ?: '-') . "</td>";
            echo "<td>" . ($user['login_ip'] ?: '-') . "</td>";
            echo "<td>" . ($user['last_login_at'] ?: '-') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
    
    echo "<div class='info'><strong>Next:</strong> Login to http://localhost/extract/login.php to see data populate</div>";
    
} catch (Exception $e) {
    echo "<div class='error'>";
    echo "<h4>❌ Error</h4>";
    echo "<p>" . $e->getMessage() . "</p>";
    echo "</div>";
}

echo "</div>";
?>
