<?php
require_once 'config/config.php';
require_once 'classes/Database.php';
require_once 'classes/Auth.php';
require_once 'classes/DeviceRestrictionChecker.php';
require_once 'classes/FaceRecognition.php';

$db = Database::getInstance();
$auth = new Auth($db);
$faceRecognition = new FaceRecognition($db);

$message = '';
$message_type = '';

if ($auth->isLoggedIn()) {
    $auth->redirectBasedOnRole();
}

// Detect if mobile device
$ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
$isMobile = preg_match('/mobile|android|iphone|windows phone|blackberry|opera mini/i', $ua);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    
    if (empty($email) || empty($password)) {
        $message = 'Please fill in all fields';
        $message_type = 'danger';
    } else {
        if ($auth->login($email, $password)) {
            // Check device restrictions before redirecting
            checkDeviceRestrictions();
            // Successful login - redirect based on role
            $auth->redirectBasedOnRole();
        } else {
            $message = 'Invalid email or password';
            $message_type = 'danger';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - <?php echo APP_NAME; ?></title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Feather Icons -->
    <script src="https://unpkg.com/feather-icons"></script>
    <!-- Face API for face detection -->
    <script src="https://cdn.jsdelivr.net/npm/face-api.js"></script>
    
    <style>
        :root {
            --primary-color: #007AFF;
            --secondary-color: #F2F2F7;
            --text-color: #1C1C1E;
            --light-gray: #F5F6F8;
            --border-color: #E5E5EA;
            --shadow-color: rgba(0, 0, 0, 0.1);
        }
        
        * {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
        }
        
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            color: var(--text-color);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .login-container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 24px;
            padding: 2.5rem;
            box-shadow: 0 20px 40px var(--shadow-color);
            border: 1px solid rgba(255, 255, 255, 0.2);
            max-width: 420px;
            width: 100%;
        }
        
        .login-header {
            text-align: center;
            margin-bottom: 2rem;
        }
        
        .login-title {
            font-size: 2rem;
            font-weight: 700;
            background: linear-gradient(135deg, #007AFF 0%, #0056CC 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 0.5rem;
        }
        
        .login-subtitle {
            color: #6C757D;
            font-size: 0.95rem;
        }
        
        .form-floating {
            margin-bottom: 1.5rem;
        }
        
        .form-control {
            border-radius: 12px;
            border: 2px solid var(--border-color);
            padding: 1rem;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: rgba(255, 255, 255, 0.8);
        }
        
        .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.25rem rgba(0, 122, 255, 0.25);
            background: white;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #007AFF 0%, #0056CC 100%);
            border: none;
            border-radius: 25px;
            padding: 12px 30px;
            font-weight: 600;
            font-size: 1rem;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(0, 122, 255, 0.3);
            width: 100%;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 122, 255, 0.4);
            background: linear-gradient(135deg, #0056CC 0%, #004499 100%);
        }
        
        .btn-secondary {
            background: linear-gradient(135deg, #34C759 0%, #00B741 100%);
            border: none;
            border-radius: 25px;
            padding: 12px 30px;
            font-weight: 600;
            font-size: 1rem;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(52, 199, 89, 0.3);
            width: 100%;
            color: white;
        }
        
        .btn-secondary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(52, 199, 89, 0.4);
        }
        
        .alert {
            border-radius: 12px;
            border: none;
            margin-bottom: 1.5rem;
        }
        
        .login-footer {
            text-align: center;
            margin-top: 2rem;
            padding-top: 1.5rem;
            border-top: 1px solid var(--border-color);
        }
        
        .login-footer a {
            color: var(--primary-color);
            text-decoration: none;
            font-weight: 500;
        }
        
        .login-footer a:hover {
            text-decoration: underline;
        }
        
        .password-toggle {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            color: #6C757D;
        }
        
        .floating-shapes {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            overflow: hidden;
            z-index: -1;
        }
        
        .shape {
            position: absolute;
            background: rgba(0, 122, 255, 0.1);
            border-radius: 50%;
            animation: float 6s ease-in-out infinite;
        }
        
        .shape:nth-child(1) {
            width: 80px;
            height: 80px;
            top: 20%;
            left: 10%;
            animation-delay: 0s;
        }
        
        .shape:nth-child(2) {
            width: 120px;
            height: 120px;
            top: 60%;
            right: 10%;
            animation-delay: 2s;
        }
        
        .shape:nth-child(3) {
            width: 60px;
            height: 60px;
            bottom: 20%;
            left: 20%;
            animation-delay: 4s;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-20px); }
        }
        
        /* Face Camera Styles */
        #face-camera-container {
            display: none;
            text-align: center;
            margin: 1.5rem 0;
        }
        
        #face-camera {
            width: 100%;
            max-width: 300px;
            border-radius: 12px;
            background: #000;
        }
        
        .face-status {
            font-size: 0.9rem;
            margin-top: 0.5rem;
            padding: 0.5rem;
            border-radius: 8px;
        }
        
        .face-status.success {
            background: rgba(52, 199, 89, 0.2);
            color: #00B741;
        }
        
        .face-status.processing {
            background: rgba(0, 122, 255, 0.2);
            color: #0056CC;
        }
        
        .face-status.error {
            background: rgba(255, 59, 48, 0.2);
            color: #FF3B30;
        }
        
        .divider {
            display: flex;
            align-items: center;
            margin: 1.5rem 0;
        }
        
        .divider::before,
        .divider::after {
            content: '';
            flex: 1;
            height: 1px;
            background: var(--border-color);
        }
        
        .divider span {
            padding: 0 0.75rem;
            color: #999;
            font-size: 0.9rem;
        }
    </style>
</head>
<body>
    <div class="floating-shapes">
        <div class="shape"></div>
        <div class="shape"></div>
        <div class="shape"></div>
    </div>
    
    <div class="login-container">
        <div class="login-header">
            <h1 class="login-title">Welcome Back</h1>
            <p class="login-subtitle">Sign in to your account</p>
        </div>
        
        <?php if ($message): ?>
        <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <!-- Face ID Option (Mobile Only) -->
        <?php if ($isMobile): ?>
        <div id="face-login-section">
            <button type="button" class="btn btn-secondary w-100 mb-3" id="start-face-btn">
                <i data-feather="camera"></i> Login with Face ID
            </button>
            
            <div id="face-camera-container">
                <video id="face-camera" autoplay playsinline></video>
                <div class="face-status" id="face-status">Initializing camera...</div>
                <button type="button" class="btn btn-sm btn-outline-secondary mt-2" id="cancel-face-btn">
                    Cancel
                </button>
            </div>
            
            <div class="divider"><span>OR</span></div>
        </div>
        <?php endif; ?>
        
        <!-- Traditional Email/Password Login -->
        <form method="POST" action="login.php" id="email-login-form">
            <div class="form-floating">
                <input type="email" class="form-control" id="email" name="email" placeholder="name@example.com" required>
                <label for="email">Email address</label>
            </div>
            
            <div class="form-floating position-relative">
                <input type="password" class="form-control" id="password" name="password" placeholder="Password" required>
                <label for="password">Password</label>
                <span class="password-toggle" onclick="togglePassword()">
                    <i data-feather="eye" id="eye-icon"></i>
                </span>
            </div>
            
            <div class="d-flex justify-content-between align-items-center mb-3">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="remember" name="remember">
                    <label class="form-check-label" for="remember">
                        Remember me
                    </label>
                </div>
                <a href="forgot-password.php" class="text-decoration-none">Forgot password?</a>
            </div>
            
            <button type="submit" class="btn btn-primary">
                <i data-feather="log-in" class="me-2"></i>
                Sign In
            </button>
        </form>
        
        <div class="login-footer">
            <?php if (ALLOW_REGISTRATION): ?>
            <p class="mb-2">Don't have an account? <a href="register.php">Sign up</a></p>
            <?php endif; ?>
            <p class="mb-0">Need help? <a href="contact.php">Contact Admin</a></p>
            <p class="mb-0">Back to <a href="index.php">Home Page</a></p>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <!-- Feather Icons -->
    <script>
        feather.replace();
        
        function togglePassword() {
            const passwordInput = document.getElementById('password');
            const eyeIcon = document.getElementById('eye-icon');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                eyeIcon.setAttribute('data-feather', 'eye-off');
            } else {
                passwordInput.type = 'password';
                eyeIcon.setAttribute('data-feather', 'eye');
            }
            feather.replace();
        }
        
        // Face ID Authentication (Mobile Only)
        const isMobile = <?php echo $isMobile ? 'true' : 'false'; ?>;
        let faceStream = null;
        let faceModel = null;
        
        if (isMobile) {
            document.getElementById('start-face-btn').addEventListener('click', startFaceLogin);
            document.getElementById('cancel-face-btn').addEventListener('click', cancelFaceLogin);
            initializeFaceAPI();
        }
        
        async function initializeFaceAPI() {
            try {
                // Load face-api models
                const MODEL_URL = 'https://cdn.jsdelivr.net/npm/@vladmandic/face-api/model/';
                await Promise.all([
                    faceapi.nets.faceRecognitionNet.loadFromUri(MODEL_URL),
                    faceapi.nets.faceLandmark68Net.loadFromUri(MODEL_URL),
                    faceapi.nets.faceDetectionNet.loadFromUri(MODEL_URL),
                    faceapi.nets.faceExpressionNet.loadFromUri(MODEL_URL),
                ]);
                console.log('Face API loaded successfully');
            } catch (e) {
                console.error('Failed to load Face API:', e);
            }
        }
        
        async function startFaceLogin() {
            const container = document.getElementById('face-camera-container');
            const video = document.getElementById('face-camera');
            const status = document.getElementById('face-status');
            
            container.style.display = 'block';
            status.textContent = 'Accessing camera...';
            status.className = 'face-status processing';
            
            try {
                faceStream = await navigator.mediaDevices.getUserMedia({
                    video: {
                        facingMode: 'user',
                        width: { ideal: 400 },
                        height: { ideal: 400 }
                    }
                });
                
                video.srcObject = faceStream;
                
                status.textContent = 'Camera ready - Position your face in the frame';
                status.className = 'face-status processing';
                
                // Start face detection loop
                detectFace(video, status);
                
            } catch (error) {
                status.textContent = 'Camera access denied or unavailable';
                status.className = 'face-status error';
                console.error('Camera error:', error);
            }
        }
        
        async function detectFace(video, status) {
            let detectionCount = 0;
            const requiredDetections = 5;
            
            const detectionInterval = setInterval(async () => {
                try {
                    const detections = await faceapi.detectAllFaces(video, new faceapi.TinyFaceDetectorOptions());
                    
                    if (detections.length > 0) {
                        detectionCount++;
                        status.textContent = `Face detected (${detectionCount}/${requiredDetections})`;
                        status.className = 'face-status processing';
                        
                        if (detectionCount >= requiredDetections) {
                            clearInterval(detectionInterval);
                            authenticateWithFace(video, status);
                        }
                    } else {
                        detectionCount = 0;
                        status.textContent = 'No face detected - Please position your face';
                        status.className = 'face-status processing';
                    }
                } catch (error) {
                    console.error('Detection error:', error);
                }
            }, 500);
        }
        
        async function authenticateWithFace(video, status) {
            status.textContent = 'Analyzing face...';
            status.className = 'face-status processing';
            
            try {
                const detections = await faceapi.detectSingleFace(video, new faceapi.TinyFaceDetectorOptions()).withFaceLandmarks().withFaceDescriptors();
                
                if (!detections) {
                    status.textContent = 'Face detection failed. Try again.';
                    status.className = 'face-status error';
                    return;
                }
                
                const faceData = JSON.stringify(detections.descriptor);
                
                // Send to server for authentication
                const response = await fetch('api/face_recognition.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'action=authenticate_face&face_data=' + encodeURIComponent(faceData)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    status.textContent = '✓ Face recognized! Logging in...';
                    status.className = 'face-status success';
                    setTimeout(() => {
                        window.location.href = 'user/dashboard.php';
                    }, 1500);
                } else {
                    status.textContent = 'Face not recognized. Try email login.';
                    status.className = 'face-status error';
                }
            } catch (error) {
                status.textContent = 'Authentication error. Try again.';
                status.className = 'face-status error';
                console.error('Auth error:', error);
            }
        }
        
        function cancelFaceLogin() {
            if (faceStream) {
                faceStream.getTracks().forEach(track => track.stop());
            }
            document.getElementById('face-camera-container').style.display = 'none';
        }
    </script>
</body>
</html>
