<?php
// user/dashboard.php (single-page AJAX dashboard) — MOBILE FRIENDLY with slide-in sidebar
// NOTE: replace the file at /extract/user/dashboard.php with this content.
error_reporting(E_ALL);
// Don't display errors - log them instead to prevent JSON corruption
ini_set('display_errors', 0);
ini_set('log_errors', 1);
file_put_contents(__DIR__ . '/debug_log.txt', "POST DATA:\n" . print_r($_POST, true) . "\n", FILE_APPEND);

require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';
require_once '../classes/User.php';
require_once '../classes/Search.php';
require_once '../classes/TelegramBot.php';
require_once '../classes/SubscriptionManager.php';
require_once '../classes/RazorpayPayment.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

if ($auth->isAdmin()) {
    header('Location: ../admin/dashboard.php');
    exit();
}

$user = new User($db);
$search = new Search($db, $user);
$subscriptionMgr = new SubscriptionManager();
$currentUser = $auth->getCurrentUser();
$telegram = new TelegramBot(TELEGRAM_BOT_TOKEN, TELEGRAM_ADMIN_CHAT_ID);

// Load subscription and trial info
$userTrial = null;
$userSubscription = null;
if ($currentUser && isset($currentUser['id'])) {
    $userTrial = $subscriptionMgr->getFreeTrial($currentUser['id']);
    $userSubscription = $subscriptionMgr->getActiveSubscription($currentUser['id']);
}

// Fetch full user data including password for verification
if ($currentUser && isset($currentUser['id'])) {
    $fullUserData = $db->fetch('SELECT * FROM users WHERE id = ?', [$currentUser['id']]);
    if ($fullUserData) {
        $currentUser['password'] = $fullUserData['password'];
    } else {
        // Fallback if user data cannot be fetched
        error_log('User data not found for ID: ' . $currentUser['id']);
    }
} else {
    error_log('Current user not properly set');
}

// Load initial data
$permissions     = $user->getPermissions($currentUser['id']);
$recentSearches  = $user->getSearchHistory($currentUser['id'], 10);
$unreadMessages  = $user->getUnreadMessages($currentUser['id']);
$adminSettings   = $search->getAdminSettings();

// ---------- AJAX/POST handlers ----------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    header('Content-Type: application/json; charset=utf-8');
    ob_start(); // Start output buffering to catch any stray output
    
    // Debug logging
    error_log('=== PAYMENT DEBUG ===');
    error_log('Action: ' . $action);
    error_log('POST Data: ' . json_encode($_POST));
    error_log('Current User: ' . json_encode($currentUser));

    switch ($action) {
        // Mobile search (AJAX)
        case 'search_mobile':
            $mobile = trim($_POST['mobile'] ?? '');
            if (!preg_match('/^[6-9]\d{9}$/', $mobile)) {
                echo json_encode(['success' => false, 'message' => 'Invalid mobile number format']);
                exit;
            }
            $cost = (float)($adminSettings['mobile_search_cost'] ?? 1);
            $result = $search->mobileSearch($currentUser['id'], $mobile);
            if ($result['success']) {
                // Send Telegram notifications
                if (!empty(TELEGRAM_BOT_TOKEN) && !empty(TELEGRAM_ADMIN_CHAT_ID)) {
                    $telegram->notifySearch($currentUser, 'mobile', $mobile, $result, $cost);
                }
                if (!empty($currentUser['telegram_id']) && !empty(TELEGRAM_BOT_TOKEN)) {
                    $telegram->notifyUserSearch($currentUser, 'mobile', $mobile, $result);
                }
                
                echo json_encode([
                    'success'  => true,
                    'data'     => $result['data'],
                    'message'  => $result['message'] ?? 'Search successful',
                    'fallback' => $result['fallback'] ?? false
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => $result['message'] ?? 'Search failed']);
            }
            exit;

        // Aadhaar search (AJAX)
        case 'search_aadhaar':
            $aadhaar = trim($_POST['aadhaar'] ?? '');
            if (!preg_match('/^\d{12}$/', $aadhaar)) {
                echo json_encode(['success' => false, 'message' => 'Invalid Aadhaar number format']);
                exit;
            }
            $cost = (float)($adminSettings['aadhaar_search_cost'] ?? 5);
            $result = $search->aadhaarSearch($currentUser['id'], $aadhaar);
            if ($result['success']) {
                // Send Telegram notifications
                if (!empty(TELEGRAM_BOT_TOKEN) && !empty(TELEGRAM_ADMIN_CHAT_ID)) {
                    $telegram->notifySearch($currentUser, 'aadhaar', $aadhaar, $result, $cost);
                }
                if (!empty($currentUser['telegram_id']) && !empty(TELEGRAM_BOT_TOKEN)) {
                    $telegram->notifyUserSearch($currentUser, 'aadhaar', $aadhaar, $result);
                }
                
                echo json_encode([
                    'success'  => true,
                    'data'     => $result['data'],
                    'message'  => $result['message'] ?? 'Search successful',
                    'fallback' => $result['fallback'] ?? false
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => $result['message'] ?? 'Search failed']);
            }
            exit;

        // Update profile (AJAX)
        case 'update_profile':
            $name  = trim($_POST['name'] ?? '');
            $email = trim($_POST['email'] ?? '');
            if ($name === '' || $email === '') {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Name and email are required.']);
                exit;
            }
            $ok = $user->updateProfile($currentUser['id'], $name, $email);
            if ($ok) {
                $_SESSION['user_name']  = $name;
                $_SESSION['user_email'] = $email;
                ob_end_clean();
                echo json_encode(['success' => true, 'message' => 'Profile updated', 'name' => $name, 'email' => $email]);
            } else {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Failed to update profile']);
            }
            exit;

        // Change password (AJAX)
        case 'change_password':
            $currentPass = $_POST['current_password'] ?? '';
            $newPass     = $_POST['new_password'] ?? '';
            $confirmPass = $_POST['confirm_password'] ?? '';

            if (empty($currentPass) || empty($newPass) || empty($confirmPass)) {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'All fields are required']);
                exit;
            }

            if ($newPass !== $confirmPass) {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'New passwords do not match']);
                exit;
            }

            if (strlen($newPass) < 6) {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Password must be at least 6 characters']);
                exit;
            }

            // Verify current password
            if (!password_verify($currentPass, $currentUser['password'])) {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Current password is incorrect']);
                exit;
            }

            // Hash new password and update
            $hashedPassword = password_hash($newPass, PASSWORD_BCRYPT, ['cost' => 12]);
            try {
                // Use query() method instead of prepare()
                $result = $db->query(
                    'UPDATE users SET password = ? WHERE id = ?',
                    [$hashedPassword, $currentUser['id']]
                );
                
                ob_end_clean();
                if ($result) {
                    echo json_encode(['success' => true, 'message' => 'Password updated successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Failed to update password']);
                }
            } catch (Exception $e) {
                error_log('Password update error: ' . $e->getMessage());
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Error updating password: ' . $e->getMessage()]);
            }
            exit;

        // Add credits (AJAX) - REMOVED: Wallet system disabled
        case 'add_credits':
            $amount = floatval($_POST['amount'] ?? 0);
            if ($amount <= 0) {
                echo json_encode(['success' => false, 'message' => 'Invalid amount']);
                exit;
            }
            // Wallet system removed - use subscription plans instead
            echo json_encode(['success' => false, 'message' => 'Wallet system has been removed. Please use subscription plans to get searches.']);
            exit;

        // Submit support ticket (AJAX)
        case 'submit_ticket':
            $title = trim($_POST['title'] ?? '');
            $body  = trim($_POST['body'] ?? '');
            if ($title === '' || $body === '') {
                echo json_encode(['success' => false, 'message' => 'Please fill all fields']);
                exit;
            }
            if (method_exists($db, 'insert')) {
                $ticketId = $db->insert('support_tickets', [
                    'user_id'    => $currentUser['id'],
                    'title'      => $title,
                    'body'       => $body,
                    'status'     => 'open',
                    'created_at' => date('Y-m-d H:i:s')
                ]);
                if ($ticketId) echo json_encode(['success' => true, 'message' => 'Ticket submitted', 'ticket_id' => $ticketId]);
                else echo json_encode(['success' => false, 'message' => 'Failed to submit ticket']);
            } else {
                $sql  = "INSERT INTO support_tickets (user_id, title, body, status, created_at) VALUES (:uid, :title, :body, 'open', NOW())";
                $stmt = $db->prepare($sql);
                $ok   = $stmt->execute([':uid' => $currentUser['id'], ':title' => $title, ':body' => $body]);
                if ($ok) echo json_encode(['success' => true, 'message' => 'Ticket submitted']);
                else echo json_encode(['success' => false, 'message' => 'Failed to submit ticket']);
            }
            exit;

        // Create payment order
        case 'create_payment_order':
            error_log('Creating payment order...');
            $amount = floatval($_POST['amount'] ?? 0);
            error_log('Amount: ' . $amount);
            
            if ($amount <= 0 || $amount > 100000) {
                error_log('Invalid amount: ' . $amount);
                echo json_encode(['success' => false, 'message' => 'Invalid amount']);
                exit;
            }
            
            $orderId = 'ORD-' . time() . '-' . $currentUser['id'];
            error_log('Generated Order ID: ' . $orderId);
            
            $result = $db->insert('payment_orders', [
                'order_id' => $orderId,
                'user_id' => $currentUser['id'],
                'amount' => $amount,
                'payment_method' => 'manual_upi',
                'status' => 'pending',
                'created_at' => date('Y-m-d H:i:s')
            ]);
            
            error_log('Insert result: ' . ($result ? 'SUCCESS' : 'FAILED'));
            
            if ($result) {
                error_log('Payment order created successfully');
                echo json_encode(['success' => true, 'order_id' => $orderId]);
            } else {
                error_log('Failed to insert payment order into database');
                echo json_encode(['success' => false, 'message' => 'Failed to create order']);
            }
            exit;

        // Submit manual payment
        case 'submit_manual_payment':
            $orderId = trim($_POST['order_id'] ?? '');
            $txnId = trim($_POST['transaction_id'] ?? '');
            
            if (empty($orderId) || empty($txnId)) {
                echo json_encode(['success' => false, 'message' => 'Missing required fields']);
                exit;
            }
            
            // Check if file uploaded
            if (empty($_FILES['payment_screenshot'])) {
                echo json_encode(['success' => false, 'message' => 'Screenshot required']);
                exit;
            }
            
            // Handle file upload
            $uploadsDir = __DIR__ . '/../uploads/payments/';
            if (!is_dir($uploadsDir)) {
                mkdir($uploadsDir, 0755, true);
            }
            
            $file = $_FILES['payment_screenshot'];
            $allowed = ['image/jpeg', 'image/png', 'image/gif'];
            
            if (!in_array($file['type'], $allowed) || $file['size'] > 5000000) {
                echo json_encode(['success' => false, 'message' => 'Invalid file or too large']);
                exit;
            }
            
            $filename = 'payment_' . $currentUser['id'] . '_' . time() . '.png';
            $filepath = $uploadsDir . $filename;
            
            if (!move_uploaded_file($file['tmp_name'], $filepath)) {
                echo json_encode(['success' => false, 'message' => 'Failed to upload file']);
                exit;
            }
            
            // Update payment order
            $screenshotPath = 'uploads/payments/' . $filename;
            $updated = $db->update('payment_orders', 
                [
                    'status' => 'awaiting_approval',
                    'transaction_id' => $txnId,
                    'screenshot_path' => $screenshotPath
                ],
                'order_id = :order_id AND user_id = :user_id',
                ['order_id' => $orderId, 'user_id' => $currentUser['id']]
            );
            
            if ($updated) {
                error_log('Payment order updated successfully');
                
                // Send Telegram notification if configured
                require_once(__DIR__ . '/../classes/Payment.php');
                $payment = new Payment();
                $paymentDetails = $payment->getPaymentDetails();
                
                error_log('Payment Details for Telegram: ' . json_encode($paymentDetails));
                
                if (!empty($paymentDetails['telegram_bot_token']) && !empty($paymentDetails['telegram_admin_chat_id'])) {
                    error_log('Sending Telegram notification...');
                    require_once(__DIR__ . '/../classes/TelegramBot.php');
                    $order = $db->fetch('SELECT * FROM payment_orders WHERE order_id = ?', [$orderId]);
                    error_log('Order found: ' . json_encode($order));
                    
                    $telegram = new TelegramBot($paymentDetails['telegram_bot_token'], $paymentDetails['telegram_admin_chat_id']);
                    $message = "💳 NEW PAYMENT SUBMITTED\n\nUser: {$currentUser['name']}\nEmail: {$currentUser['email']}\nAmount: ₹{$order['amount']}\nUTR: {$txnId}";
                    error_log('Sending message: ' . $message);
                    
                    $telegram->sendApprovalButtons(
                        $paymentDetails['telegram_admin_chat_id'],
                        $message,
                        $orderId
                    );
                    error_log('Telegram notification sent');
                } else {
                    error_log('Telegram not configured - token or chat_id is empty');
                }
                
                echo json_encode(['success' => true, 'message' => 'Payment submitted successfully']);
            } else {
                error_log('Failed to update payment order');
                echo json_encode(['success' => false, 'message' => 'Failed to process payment']);
            }
            exit;

        // Clear all search history
        case 'clear_search_history':
            try {
                $result = $db->query(
                    'DELETE FROM search_logs WHERE user_id = ?',
                    [$currentUser['id']]
                );
                if ($result !== false) {
                    ob_end_clean();
                    echo json_encode(['success' => true, 'message' => 'All search history cleared successfully']);
                } else {
                    ob_end_clean();
                    echo json_encode(['success' => false, 'message' => 'Failed to clear search history']);
                }
            } catch (Exception $e) {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Error clearing search history: ' . $e->getMessage()]);
            }
            exit;

        // Clear search history by type (mobile or aadhaar)
        case 'clear_search_history_by_type':
            try {
                $type = trim($_POST['type'] ?? '');
                if (!in_array($type, ['mobile', 'aadhaar'])) {
                    ob_end_clean();
                    echo json_encode(['success' => false, 'message' => 'Invalid search type']);
                    exit;
                }
                
                $result = $db->query(
                    'DELETE FROM search_logs WHERE user_id = ? AND type = ?',
                    [$currentUser['id'], $type]
                );
                if ($result !== false) {
                    ob_end_clean();
                    echo json_encode(['success' => true, 'message' => 'All ' . $type . ' searches cleared successfully']);
                } else {
                    ob_end_clean();
                    echo json_encode(['success' => false, 'message' => 'Failed to clear search history']);
                }
            } catch (Exception $e) {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Error clearing search history: ' . $e->getMessage()]);
            }
            exit;

        // Get user's transactions (purchased plans)
        case 'get_my_transactions':
            try {
                $transactions = $db->fetchAll(
                    "SELECT us.id, us.user_id, us.plan_id, us.status, us.payment_status, us.start_date, us.end_date, us.search_used, us.created_at,
                            us.razorpay_order_id, us.razorpay_payment_id,
                            sp.plan_name, sp.duration_days, sp.price, sp.search_limit
                     FROM user_subscriptions us 
                     LEFT JOIN subscription_plans sp ON us.plan_id = sp.id 
                     WHERE us.user_id = ? 
                     ORDER BY us.created_at DESC",
                    [$currentUser['id']]
                );
                
                ob_end_clean();
                echo json_encode([
                    'success' => true,
                    'transactions' => $transactions ?: []
                ]);
            } catch (Exception $e) {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
            }
            exit;

        // Get active subscription status (for real-time checking)
        case 'check_subscription_status':
            try {
                $activeSubscription = $subscriptionMgr->getActiveSubscription($currentUser['id']);
                $userTrial = $subscriptionMgr->getFreeTrial($currentUser['id']);
                
                ob_end_clean();
                echo json_encode([
                    'success' => true,
                    'subscription' => $activeSubscription,
                    'trial' => $userTrial ? [
                        'is_active' => $userTrial['is_active'],
                        'searches_remaining' => $userTrial['searches_remaining']
                    ] : null
                ]);
            } catch (Exception $e) {
                ob_end_clean();
                echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
            }
            exit;

        default:
            echo json_encode(['success' => false, 'message' => 'Unknown action']);
            exit;
    }
}
// ---------- End AJAX handlers ----------
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Dashboard - <?php echo APP_NAME; ?></title>

<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<script src="https://unpkg.com/feather-icons"></script>
<script src="https://checkout.razorpay.com/v1/checkout.js"></script>

<style>
:root {--primary-color:#007AFF;--secondary-color:#F2F2F7;--text-color:#1C1C1E;--light-gray:#F5F6F8;--border-color:#E5E5EA;--shadow-color:rgba(0,0,0,0.1);}
*{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,"Helvetica Neue",Arial,sans-serif;box-sizing:border-box;}
body{background:var(--light-gray);color:var(--text-color);margin:0}

/* === Off-canvas Sidebar (slide-in) === */
.sidebar{
  position:fixed; top:0; left:0; height:100vh; width:260px;
  background:#fff; border-right:1px solid var(--border-color);
  box-shadow:0 4px 20px var(--shadow-color);
  z-index:1040; padding-bottom:20px; overflow-y:auto;
  transform:translateX(0); transition:transform .3s ease;
}
#premiumSidebar{
  left:auto; right:0; width:300px; background:linear-gradient(135deg, #667eea 0%, #764ba2 100%);
  border-right:0; border-left:1px solid rgba(255,255,255,.2);
  transform:translateX(100%); transition:transform .3s ease;
}
.sidebar-header{padding:1.25rem 1rem;border-bottom:1px solid var(--border-color);text-align:center}
.sidebar-menu{list-style:none;margin:0;padding:0}
.sidebar-menu a{display:flex;align-items:center;gap:.75rem;padding:0.9rem 1rem;color:var(--text-color);text-decoration:none;border-left:3px solid transparent;transition:all .2s}
.sidebar-menu a:hover,.sidebar-menu a.active{background:rgba(0,122,255,.08);border-left-color:var(--primary-color);color:var(--primary-color)}
.sidebar-menu i{width:20px;height:20px}

/* Overlay for mobile when sidebar is open */
.overlay{
  display:none; position:fixed; inset:0; background:rgba(0,0,0,.35); z-index:1035;
}

/* Main content */
.main-content{
  margin-left:260px; padding:16px; transition:margin-left .3s ease;
}
.top-bar{
  background: linear-gradient(135deg, #007AFF 0%, #0056CC 100%);
  border-radius:16px;
  padding:1.5rem 1.5rem;
  margin-bottom:1.5rem;
  box-shadow:0 8px 24px rgba(0,122,255,.2);
  display:flex;
  align-items:center;
  justify-content:space-between;
  gap:15px;
  color: white;
}
.top-bar h4 {
  color: white !important;
}
.top-bar .text-muted {
  color: rgba(255,255,255,0.8) !important;
}
.wallet-badge{
  background:linear-gradient(135deg,#007AFF 0%,#0056CC 100%); 
  color:#fff;
  padding:.6rem 1.2rem;
  border-radius:20px;
  font-weight:600;
  font-size:.85rem;
  box-shadow:0 4px 12px rgba(0,122,255,.25);
  border: none;
  display:inline-flex;
  align-items:center;
  gap:6px;
}
.card{
  background:#fff;
  border:none;
  border-radius:16px;
  box-shadow:0 4px 16px rgba(0,0,0,.08);
  margin-bottom:1.5rem;
  transition:all .3s ease;
  overflow: hidden;
}
.card:hover{
  box-shadow:0 8px 24px rgba(0,0,0,.12);
  transform:translateY(-2px);
}
.card-header{
  background:linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
  border-bottom:1px solid var(--border-color);
  padding:1.25rem;
  border-radius:16px 16px 0 0;
  font-weight:600;
  color: #1C1C1E;
}
.card-body{padding:1.25rem; font-size: 0.95rem;}
.btn-primary{
  background:linear-gradient(135deg,#007AFF 0%,#0056CC 100%);
  border:none;
  border-radius:12px;
  padding:10px 16px;
  font-weight:600;
  transition:all .2s;
}
.btn-primary:hover{
  transform:translateY(-2px);
  box-shadow:0 8px 22px rgba(0,122,255,.35);
}
.btn-primary:active {
  transform: translateY(0);
}
.form-control{
  border-radius:12px;
  border:2px solid var(--border-color);
  padding:.75rem 1rem;
  transition: all .2s;
}
.form-control:focus{
  border-color:var(--primary-color);
  box-shadow:0 0 0 .25rem rgba(0,122,255,.2);
  outline: none;
}
.table {
  font-size: 0.9rem;
  margin-bottom: 0;
}
.table thead {
  background: linear-gradient(135deg, #f0f4ff 0%, #f8f9fa 100%);
  font-weight: 600;
  color: #1C1C1E;
}
.table th {
  border-color: var(--border-color);
  padding: 0.75rem;
}
.table td {
  border-color: var(--border-color);
  padding: 0.75rem;
  vertical-align: middle;
}
.table tbody tr:hover {
  background-color: #f8f9fa;
}
.badge {
  font-size: 0.75rem;
  padding: 0.35rem 0.6rem;
  font-weight: 600;
}
.section{display:none}
.section.active{display:block}
.table-responsive{overflow-x:auto}
.small-muted{color:#6c757d;font-size:.9rem}

/* Sidebar toggle button */
.sidebar-toggle{
  display:none; border:none; background:#fff; border-radius:12px; padding:.4rem .6rem;
  box-shadow:0 4px 12px rgba(0,0,0,.08);
}

/* === Responsive === */
@media (max-width: 991px){
  .sidebar{transform:translateX(-100%);}        /* hidden by default on mobile */
  .sidebar.open{transform:translateX(0);}        /* visible when toggled */
  .overlay.show{display:block;}
  .main-content{margin-left:0; padding:14px}
  .sidebar-toggle{display:inline-flex; align-items:center; justify-content:center}
  .top-bar{gap:8px}
  .wallet-badge{font-size:.8rem;padding:.4rem .7rem}
  .btn, .btn-primary{width:100%}
  .text-end{text-align:left !important}
}
@media (max-width: 576px){
  .top-bar small.text-muted{display:none}
}
</style>
</head>
<body>

<!-- Mobile overlay -->
<div class="overlay" id="pageOverlay"></div>

<!-- Sidebar Toggle (in top bar; rendered later with JS icons) -->
<!-- We place it before main so it's focusable -->
<button class="sidebar-toggle" id="sidebarToggle" aria-label="Open menu" style="position:fixed;top:12px;left:12px;z-index:1050;">
  <i data-feather="menu"></i>
</button>

<!-- Sidebar -->
<aside class="sidebar" id="sidebar">
  <div class="sidebar-header">
    <div style="display: flex; align-items: center; gap: 6px;">
      <h5 class="mb-0" id="user-name"><?php echo htmlspecialchars($currentUser['name']); ?></h5>
      <!-- Instagram Style Blue Checkmark Badge -->
      <svg width="22" height="22" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" title="Verified" style="margin-top: 3px; filter: drop-shadow(0 0 2px rgba(0,0,0,0.1));">
        <circle cx="12" cy="12" r="11" fill="#1D9BF0"/>
        <circle cx="12" cy="12" r="11" fill="none" stroke="#1D9BF0" stroke-width="1.5" opacity="0.3"/>
        <path d="M8.5 12L10.5 14L15.5 9" stroke="white" stroke-width="2.5" fill="none" stroke-linecap="round" stroke-linejoin="round"/>
      </svg>
    </div>
    <small class="text-muted">User Dashboard</small>
  </div>

  <!-- Wallet Sidebar Widget -->
  <!-- REMOVED: Wallet system disabled -->
  <!-- <?php include(__DIR__ . '/wallet_sidebar.php'); ?> -->

  <ul class="sidebar-menu">
    <li><a href="#" data-section="dashboard" class="nav-link active" onclick="showSection('dashboard')"><i data-feather="home"></i> Dashboard</a></li>
    <li><a href="#" data-section="mobile" class="nav-link" onclick="showSection('mobile')"><i data-feather="search"></i> Mobile Search</a></li>
    <li><a href="#" data-section="aadhaar" class="nav-link" onclick="showSection('aadhaar')"><i data-feather="credit-card"></i> Aadhaar Search</a></li>
    <li><a href="#" class="nav-link" onclick="togglePremiumSidebar(event)"><i data-feather="star"></i> <strong>Premium</strong> <span class="badge bg-warning ms-auto">✨</span></a></li>
    <li><a href="#" data-section="transactions" class="nav-link" onclick="showSection('transactions')"><i data-feather="shopping-cart"></i> My Transactions</a></li>
    <li><a href="#" data-section="analytics" class="nav-link" onclick="showSection('analytics')"><i data-feather="bar-chart-2"></i> Analytics</a></li>
    <li><a href="#" data-section="profile" class="nav-link" onclick="showSection('profile')"><i data-feather="user"></i> Profile</a></li>
    <li><a href="#" data-section="support" class="nav-link" onclick="showSection('support')"><i data-feather="message-circle"></i> Support <?php if (!empty($unreadMessages)): ?><span class="badge bg-danger ms-auto" id="unread-count"><?php echo count($unreadMessages); ?></span><?php endif; ?></a></li>
    <li><a href="chatbot.php" class="nav-link"><i data-feather="message-square"></i> 💬 Chat Admin</a></li>
    <li><a href="#" data-section="face_id" class="nav-link" onclick="showSection('face_id')"><i data-feather="camera"></i> 📸 Face ID</a></li>
    <li><a href="../logout.php"><i data-feather="log-out"></i> Logout</a></li>
  </ul>
</aside>

<!-- Premium Sidebar Modal -->
<div id="premiumSidebar" class="sidebar" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); z-index: 1051;">
  <div class="sidebar-header" style="border-bottom-color: rgba(255,255,255,0.2); color: white;">
    <h5 class="mb-0">💎 Premium Plans</h5>
    <small style="color: rgba(255,255,255,0.9);">Upgrade your account</small>
  </div>
  
  <div style="padding: 20px; overflow-y: auto; height: calc(100vh - 120px);">
    <!-- Active Subscription Info -->
    <div id="activeSubInfo" style="display: none; background: rgba(255,255,255,0.15); padding: 15px; border-radius: 8px; margin-bottom: 20px; color: white;">
      <div style="font-size: 12px; opacity: 0.9; margin-bottom: 5px;">✅ Active Subscription</div>
      <div id="activeSubName" style="font-size: 16px; font-weight: bold; margin-bottom: 3px;"></div>
      <div id="activeSubDays" style="font-size: 12px; opacity: 0.85;"></div>
      <div id="activeSubSearches" style="font-size: 12px; opacity: 0.85; margin-top: 5px;"></div>
    </div>
    
    <!-- Trial Info -->
    <div id="trialInfo" style="display: none; background: rgba(255,165,0,0.2); padding: 15px; border-radius: 8px; margin-bottom: 20px; color: white;">
      <div style="font-size: 12px; opacity: 0.9; margin-bottom: 5px;">⏳ Free Trial Active</div>
      <div style="font-size: 14px; font-weight: bold; margin-bottom: 5px;">10 Searches</div>
      <div id="trialUsed" style="font-size: 12px; opacity: 0.85;"></div>
      <button onclick="togglePremiumSidebar()" style="margin-top: 10px; width: 100%; padding: 8px; background: rgba(255,255,255,0.3); border: 1px solid white; color: white; border-radius: 4px; cursor: pointer; font-weight: bold;">
        Upgrade to Premium
      </button>
    </div>
    
    <!-- Plans Grid -->
    <div id="plansContainer" style="display: grid; gap: 15px;">
      <p style="text-align: center; color: white; opacity: 0.8;">Loading plans...</p>
    </div>
    
    <hr style="border-color: rgba(255,255,255,0.2); margin: 20px 0;">
    
    <!-- Close Button -->
    <button onclick="togglePremiumSidebar()" style="width: 100%; padding: 10px; background: rgba(255,255,255,0.2); border: 1px solid white; color: white; border-radius: 4px; cursor: pointer; font-weight: bold;">
      ✕ Close
    </button>
  </div>
</div>

<!-- Overlay for premium sidebar -->
<div id="premiumOverlay" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); z-index: 1050; cursor: pointer;" onclick="togglePremiumSidebar()"></div>

<!-- Main Content -->
<main class="main-content" id="mainContent">
  <div class="top-bar">
    <div class="d-flex align-items-center gap-2" style="flex: 1;">
      <!-- show hamburger only on mobile (CSS handles visibility) -->
      <button class="sidebar-toggle d-inline-flex d-lg-none" id="sidebarToggleTop" aria-label="Open menu">
        <i data-feather="menu"></i>
      </button>
      <div>
        <h4 class="mb-0" style="font-weight: 700; color: #1C1C1E;">Welcome, <?php echo htmlspecialchars(explode(' ', $currentUser['name'])[0]); ?></h4>
        <small class="text-muted">Your Dashboard</small>
      </div>
    </div>
    <div class="d-flex align-items-center gap-3" style="flex-wrap: wrap; justify-content: flex-end;">
      <!-- Subscription Status Badge -->
      <?php if ($userSubscription && $userSubscription['status'] === 'active'): ?>
        <span class="wallet-badge" style="background: linear-gradient(135deg, #34C759 0%, #00A86B 100%); font-size: 0.85rem;">
          <i data-feather="star" style="width:14px;height:14px;" class="me-1"></i>
          Premium
        </span>
      <?php elseif ($userTrial && $userTrial['is_active']): ?>
        <span class="wallet-badge" style="background: linear-gradient(135deg, #FF9500 0%, #FF7A00 100%); font-size: 0.85rem;">
          <i data-feather="gift" style="width:14px;height:14px;" class="me-1"></i>
          Trial Mode
        </span>
      <?php else: ?>
        <span class="wallet-badge" style="background: linear-gradient(135deg, #FF3B30 0%, #CC0000 100%); font-size: 0.85rem;">
          <i data-feather="alert-circle" style="width:14px;height:14px;" class="me-1"></i>
          <a href="subscribe.php" style="color: white; text-decoration: none;">Upgrade Now</a>
        </span>
      <?php endif; ?>
    </div>
  </div>

  <!-- ALERT PLACEHOLDER -->
  <div id="global-alert"></div>

  <!-- SECTION: Dashboard / Welcome -->
  <div id="dashboard-section" class="section active">
    <!-- All Recent Searches -->
    <div class="card">
      <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">All Recent Searches</h5>
        <?php if (!empty($recentSearches)): ?>
          <button type="button" class="btn btn-sm btn-outline-danger" onclick="clearSearchHistory()">
            <i data-feather="trash-2" style="width: 16px; height: 16px;"></i> Clear All
          </button>
        <?php endif; ?>
      </div>
      <div class="card-body">
        <?php if (empty($recentSearches)): ?>
          <p class="text-muted">No searches yet</p>
        <?php else: ?>
          <div class="table-responsive">
            <table class="table table-hover table-striped">
              <thead class="table-light">
                <tr>
                  <th>Type</th>
                  <th>Query</th>
                  <th>Cost</th>
                  <th>Status</th>
                  <th>Date</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($recentSearches as $s): ?>
                  <tr>
                    <td>
                      <span class="badge bg-<?php echo $s['type'] === 'mobile' ? 'primary' : 'success'; ?>">
                        <?php echo strtoupper($s['type']); ?>
                      </span>
                    </td>
                    <td>
                      <strong><?php echo htmlspecialchars($s['query']); ?></strong>
                    </td>
                    <td>
                      <span class="text-muted">₹<?php echo number_format($s['cost'], 1); ?></span>
                    </td>
                    <td>
                      <span class="badge bg-<?php echo $s['status'] === 'success' ? 'success' : 'danger'; ?>">
                        <?php echo strtoupper($s['status']); ?>
                      </span>
                    </td>
                    <td>
                      <small class="text-muted"><?php echo date('M j, Y H:i', strtotime($s['created_at'])); ?></small>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <!-- SECTION: Mobile Search -->
  <div id="mobile-section" class="section">
    <div class="card">
      <div class="card-header"><h5>Mobile Lookup</h5></div>
      <div class="card-body">
        <form id="mobile-search-form" class="row g-3">
          <div class="col-md-8">
            <div style="position: relative;">
              <input type="text" id="mobile-input" class="form-control" placeholder="Enter 10-digit mobile number" pattern="[6-9]\d{9}" required autocomplete="off">
              <div id="mobile-suggestions" class="suggestions-dropdown" style="display: none; position: absolute; top: 100%; left: 0; right: 0; background: white; border: 1px solid #ddd; border-top: none; border-radius: 0 0 4px 4px; max-height: 200px; overflow-y: auto; z-index: 1000; box-shadow: 0 2px 4px rgba(0,0,0,0.1);"></div>
            </div>
          </div>
          <div class="col-md-4">
            <button type="submit" class="btn btn-primary w-100">Search</button>
          </div>
        </form>
        <div id="mobile-result" class="mt-4"></div>

        <!-- Recent Mobile Searches -->
        <hr>
        <div class="d-flex justify-content-between align-items-center mt-4 mb-3">
          <h6 class="mb-0">Recent Mobile Searches</h6>
          <?php 
            $mobilSearches = array_filter($recentSearches, fn($s) => $s['type'] === 'mobile');
            if (!empty($mobilSearches)): 
          ?>
            <button type="button" class="btn btn-sm btn-outline-danger" onclick="clearSearchType('mobile')">
              <i data-feather="trash-2" style="width: 14px; height: 14px;"></i> Clear
            </button>
          <?php endif; ?>
        </div>
        <div id="recent-mobile-searches" class="mt-3">
          <?php 
            if (empty($mobilSearches)): 
          ?>
            <p class="text-muted small">No mobile searches yet</p>
          <?php else: ?>
            <div class="list-group">
              <?php foreach ($mobilSearches as $s): ?>
                <button type="button" class="list-group-item list-group-item-action" onclick="searchMobileFromHistory('<?php echo htmlspecialchars($s['query']); ?>')">
                  <div class="d-flex w-100 justify-content-between align-items-center">
                    <strong><?php echo htmlspecialchars($s['query']); ?></strong>
                    <small><span class="badge bg-<?php echo $s['status'] === 'success' ? 'success' : 'danger'; ?>"><?php echo strtoupper($s['status']); ?></span></small>
                  </div>
                  <small class="text-muted"><?php echo date('M j, Y H:i', strtotime($s['created_at'])); ?></small>
                </button>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>

  <!-- SECTION: Aadhaar Search -->
  <div id="aadhaar-section" class="section">
    <div class="card">
      <div class="card-header"><h5>Aadhaar Lookup</h5></div>
      <div class="card-body">
        <form id="aadhaar-search-form" class="row g-3">
          <div class="col-md-8">
            <div style="position: relative;">
              <input type="text" id="aadhaar-input" class="form-control" placeholder="Enter 12-digit Aadhaar number" pattern="\d{12}" required autocomplete="off">
              <div id="aadhaar-suggestions" class="suggestions-dropdown" style="display: none; position: absolute; top: 100%; left: 0; right: 0; background: white; border: 1px solid #ddd; border-top: none; border-radius: 0 0 4px 4px; max-height: 200px; overflow-y: auto; z-index: 1000; box-shadow: 0 2px 4px rgba(0,0,0,0.1);"></div>
            </div>
          </div>
          <div class="col-md-4">
            <button type="submit" class="btn btn-primary w-100">Search</button>
          </div>
        </form>
        <div id="aadhaar-result" class="mt-4"></div>

        <!-- Recent Aadhaar Searches -->
        <hr>
        <div class="d-flex justify-content-between align-items-center mt-4 mb-3">
          <h6 class="mb-0">Recent Aadhaar Searches</h6>
          <?php 
            $aadhaarSearches = array_filter($recentSearches, fn($s) => $s['type'] === 'aadhaar');
            if (!empty($aadhaarSearches)): 
          ?>
            <button type="button" class="btn btn-sm btn-outline-danger" onclick="clearSearchType('aadhaar')">
              <i data-feather="trash-2" style="width: 14px; height: 14px;"></i> Clear
            </button>
          <?php endif; ?>
        </div>
        <div id="recent-aadhaar-searches" class="mt-3">
          <?php 
            if (empty($aadhaarSearches)): 
          ?>
            <p class="text-muted small">No Aadhaar searches yet</p>
          <?php else: ?>
            <div class="list-group">
              <?php foreach ($aadhaarSearches as $s): ?>
                <button type="button" class="list-group-item list-group-item-action" onclick="searchAadhaarFromHistory('<?php echo htmlspecialchars($s['query']); ?>')">
                  <div class="d-flex w-100 justify-content-between align-items-center">
                    <strong><?php echo htmlspecialchars($s['query']); ?></strong>
                    <small><span class="badge bg-<?php echo $s['status'] === 'success' ? 'success' : 'danger'; ?>"><?php echo strtoupper($s['status']); ?></span></small>
                  </div>
                  <small class="text-muted"><?php echo date('M j, Y H:i', strtotime($s['created_at'])); ?></small>
                </button>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>

  <!-- SECTION: Wallet -->
  <!-- SECTION: Transactions -->
  <div id="transactions-section" class="section">
    <div class="card">
      <div class="card-header"><h5>💳 My Transactions & Plans</h5></div>
      <div class="card-body">
        <!-- Available Plans -->
        <h6 class="mb-3">📦 Available Subscription Plans</h6>
        <div id="available-plans-container" class="row g-3 mb-4">
          <p class="text-muted">Loading plans...</p>
        </div>

        <hr>

        <!-- My Purchases -->
        <h6 class="mb-3">✅ My Purchased Plans</h6>
        <div class="table-responsive">
          <table class="table table-hover table-sm">
            <thead style="background: #f5f5f5;">
              <tr>
                <th>Plan Name</th>
                <th>Duration</th>
                <th>Amount</th>
                <th>Order ID</th>
                <th>Payment ID</th>
                <th>Status</th>
                <th>Expires</th>
                <th>Action</th>
              </tr>
            </thead>
            <tbody id="purchased-plans-tbody">
              <tr><td colspan="8" class="text-muted text-center">Loading...</td></tr>
            </tbody>
          </table>
        </div>
      </div>
    </div>
  </div>

  <!-- SECTION: Profile -->
  <div id="profile-section" class="section">
    <div class="card">
      <div class="card-header"><h5>Profile</h5></div>
      <div class="card-body">
        <form id="profile-form">
          <div class="mb-3">
            <label class="form-label">Full Name</label>
            <input type="text" name="name" id="profile-name" class="form-control" required value="<?php echo htmlspecialchars($currentUser['name']); ?>">
          </div>
          <div class="mb-3">
            <label class="form-label">Email</label>
            <div class="input-group">
              <input type="email" name="email" id="profile-email" class="form-control" required value="<?php echo htmlspecialchars($currentUser['email']); ?>">
              <span class="input-group-text">
                <?php 
                  $userVerified = $db->fetch('SELECT email_verified FROM users WHERE id = ?', [$currentUser['id']]);
                  if ($userVerified && $userVerified['email_verified']): 
                ?>
                  <span class="badge bg-primary" style="display: flex; align-items: center; gap: 5px; padding: 6px 10px; border-radius: 20px; font-size: 12px;">
                    <svg width="16" height="16" viewBox="0 0 40 40" fill="currentColor" xmlns="http://www.w3.org/2000/svg">
                      <path d="M19.998 3.094 14.638 0l-2.972 5.15H5.432v6.354L0 14.64 3.094 20 0 25.359l5.432 3.137v5.905h5.975L14.638 40l5.36-3.094L25.358 40l3.232-5.6h6.162v-6.01L40 25.359 36.905 20 40 14.641l-5.248-3.03v-6.46h-6.419L25.358 0l-5.36 3.094Zm7.415 11.225 2.254 2.287-11.43 11.5-6.835-6.93 2.244-2.258 4.587 4.581 9.18-9.18Z" fill-rule="evenodd"/>
                    </svg>
                    Verified
                  </span>
                <?php else: ?>
                  <span class="badge bg-warning text-dark" style="display: flex; align-items: center; gap: 5px; padding: 6px 10px; border-radius: 20px; font-size: 12px;">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" xmlns="http://www.w3.org/2000/svg">
                      <circle cx="12" cy="12" r="10"></circle>
                      <line x1="12" y1="8" x2="12" y2="12"></line>
                      <line x1="12" y1="16" x2="12.01" y2="16"></line>
                    </svg>
                    Not Verified
                  </span>
                <?php endif; ?>
              </span>
            </div>
            <?php if (!($userVerified && $userVerified['email_verified'])): ?>
              <small class="text-warning d-block mt-2">
                <a href="../verify_email.php?name=<?php echo urlencode($currentUser['name']); ?>" class="text-warning" style="text-decoration: none;">➜ Click here to verify your email</a>
              </small>
            <?php endif; ?>
          </div>
          <div class="d-flex gap-2 justify-content-end">
            <button type="submit" class="btn btn-primary">Save Profile</button>
            <button type="button" class="btn btn-warning" data-bs-toggle="modal" data-bs-target="#passwordModal">Change Password</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- SECTION: Support -->
  <div id="support-section" class="section">
    <div class="card">
      <div class="card-header"><h5>💬 Support Chat</h5></div>
      <div class="card-body">
        <p class="text-muted mb-3">Need help? Chat directly with our admin team in real-time.</p>
        <a href="support.php" class="btn btn-primary btn-lg w-100">
          <i data-feather="message-circle"></i> Open Chat
        </a>
      </div>
    </div>
  </div>

  <!-- SECTION: AI Analytics -->
  <div id="analytics-section" class="section">
    <div class="card">
      <div class="card-header"><h5>📊 Analytics & Insights</h5></div>
      <div class="card-body">
        <div class="row">
          <!-- Total Searches -->
          <div class="col-md-4 mb-3">
            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px; text-align: center;">
              <div style="font-size: 32px; font-weight: bold; margin-bottom: 5px;" id="total-searches">0</div>
              <div style="font-size: 14px; opacity: 0.9;">Total Searches</div>
            </div>
          </div>
          
          <!-- Mobile Searches -->
          <div class="col-md-4 mb-3">
            <div style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white; padding: 20px; border-radius: 8px; text-align: center;">
              <div style="font-size: 32px; font-weight: bold; margin-bottom: 5px;" id="mobile-searches">0</div>
              <div style="font-size: 14px; opacity: 0.9;">📱 Mobile Searches</div>
            </div>
          </div>
          
          <!-- Aadhaar Searches -->
          <div class="col-md-4 mb-3">
            <div style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); color: white; padding: 20px; border-radius: 8px; text-align: center;">
              <div style="font-size: 32px; font-weight: bold; margin-bottom: 5px;" id="aadhaar-searches">0</div>
              <div style="font-size: 14px; opacity: 0.9;">🆔 Aadhaar Searches</div>
            </div>
          </div>
        </div>
        
        <hr>
        
        <div class="row">
          <!-- Searches Remaining -->
          <div class="col-md-6 mb-3">
            <div style="background: #d4edda; padding: 15px; border-radius: 8px; border-left: 4px solid #28a745;">
              <div style="font-size: 12px; color: #155724; margin-bottom: 5px;">🔍 Searches Remaining</div>
              <div style="font-size: 24px; font-weight: bold; color: #333;" id="current-balance">0</div>
              <div style="font-size: 11px; color: #666; margin-top: 3px;" id="plan-info"></div>
            </div>
          </div>
          
          <!-- Success Rate -->
          <div class="col-md-6 mb-3">
            <div style="background: #d1ecf1; padding: 15px; border-radius: 8px; border-left: 4px solid #17a2b8;">
              <div style="font-size: 12px; color: #0c5460; margin-bottom: 5px;">✅ Success Rate</div>
              <div style="font-size: 24px; font-weight: bold; color: #333;"><span id="success-rate">0</span>%</div>
            </div>
          </div>
        </div>
        
        <hr>
        
        <!-- Account Status -->
        <div class="row">
          <div class="col-12 mb-3">
            <div style="background: #e2e3e5; padding: 15px; border-radius: 8px; border-left: 4px solid #6c757d;">
              <div style="font-size: 12px; color: #383d41; margin-bottom: 5px;">🟢 Account Status</div>
              <div style="font-size: 18px; font-weight: bold; color: #28a745;">Active</div>
            </div>
          </div>
        </div>
        
        <hr>
        
        <!-- Recent Activity -->
        <h6 class="mt-4 mb-3">📈 Recent Search Activity</h6>
        <div id="recent-activity" style="font-size: 13px; color: #666;">
          <p class="text-muted">Loading activity data...</p>
        </div>
        
        <hr>
        
        <button onclick="loadAnalytics()" class="btn btn-sm btn-outline-primary">
          <i data-feather="refresh-cw" style="width: 14px; height: 14px;"></i> Refresh Data
        </button>
      </div>
    </div>
  </div>

  <!-- SECTION: Face ID -->
  <div id="face_id-section" class="section">
    <div class="card">
      <div class="card-header"><h5>📸 Face ID Setup</h5></div>
      <div class="card-body">
        <p class="text-muted mb-3">Register your face for quick and secure one-time authentication. After registration, you can log in using Face ID from the login page.</p>
        
        <div id="face-status" style="margin-bottom: 20px;">
          <div class="alert alert-info">
            <strong>Status:</strong> <span id="face-status-text">Checking...</span>
          </div>
        </div>

        <!-- Registration Form -->
        <div id="face-registration-form" style="display: none;">
          <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
            <h6 class="mb-3">📷 Capture Your Face</h6>
            <p style="font-size: 13px; color: #666; margin-bottom: 15px;">
              Allow camera access and position your face clearly in the frame. Click "Capture Face" when ready.
            </p>
            
            <!-- Video Preview -->
            <div style="position: relative; background: #000; border-radius: 8px; overflow: hidden; margin-bottom: 15px; aspect-ratio: 4/3;">
              <video id="face-video" autoplay playsinline style="width: 100%; height: 100%; display: block;"></video>
              <canvas id="face-canvas" style="display: none;"></canvas>
              <div style="position: absolute; bottom: 10px; right: 10px; background: rgba(0,0,0,0.5); color: white; padding: 5px 10px; border-radius: 4px; font-size: 12px;" id="face-countdown">
                Ready
              </div>
            </div>

            <!-- Buttons -->
            <div class="d-flex gap-2 flex-wrap">
              <button type="button" class="btn btn-primary" onclick="startFaceCamera()">
                <i data-feather="camera"></i> Start Camera
              </button>
              <button type="button" class="btn btn-success" id="capture-face-btn" onclick="captureFace()" style="display: none;">
                <i data-feather="check-circle"></i> Capture Face
              </button>
              <button type="button" class="btn btn-secondary" onclick="stopFaceCamera()">
                <i data-feather="x-circle"></i> Close Camera
              </button>
            </div>

            <!-- Messages -->
            <div id="face-messages" style="margin-top: 15px;"></div>
          </div>
        </div>

        <!-- Delete Form (shown if face is registered) -->
        <div id="face-delete-form" style="display: none;">
          <div style="background: #fff3cd; padding: 15px; border-radius: 8px; margin-bottom: 15px; border-left: 4px solid #ffc107;">
            <strong>✓ Face Registered</strong>
            <p style="margin-top: 10px; margin-bottom: 0; font-size: 14px;">Your face is registered and ready to use for login.</p>
          </div>
          <button type="button" class="btn btn-danger" onclick="deleteFaceData()">
            <i data-feather="trash-2"></i> Delete Face Registration
          </button>
        </div>

        <hr>
        
        <h6 class="mb-2">ℹ️ How Face ID Works</h6>
        <ul style="font-size: 13px; color: #666;">
          <li>Register your face once in this dashboard</li>
          <li>Face data is stored securely and never shared</li>
          <li>You can log in using Face ID from the login page</li>
          <li>You can delete and re-register your face anytime</li>
          <li>Email and password login remains available as backup</li>
        </ul>
      </div>
    </div>
  </div>

</main>

<!-- scripts -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://checkout.razorpay.com/v1/checkout.js"></script>
<script>
feather.replace();

/* ======= GLOBAL VARIABLES ======= */
let currentOrderId = null;
let currentAmount = null;

/* ======= Sidebar toggle (slide-in) ======= */
const sidebar        = document.getElementById('sidebar');
const overlay        = document.getElementById('pageOverlay');
const sidebarToggle  = document.getElementById('sidebarToggle');
const sidebarToggleTop = document.getElementById('sidebarToggleTop');

function openSidebar(){
  if (sidebar) {
    sidebar.classList.add('open');
  }
  if (overlay) {
    overlay.classList.add('show');
  }
  document.body.style.overflow = 'hidden';
}
function closeSidebar(){
  if (sidebar) {
    sidebar.classList.remove('open');
  }
  if (overlay) {
    overlay.classList.remove('show');
  }
  document.body.style.overflow = '';
}

// Safely attach event listeners
if (sidebarToggle) {
  sidebarToggle.addEventListener('click', (e)=>{ e.preventDefault(); openSidebar(); });
}
if (sidebarToggleTop) {
  sidebarToggleTop.addEventListener('click', (e)=>{ e.preventDefault(); openSidebar(); });
}
if (overlay) {
  overlay.addEventListener('click', closeSidebar);
}
// Close when a nav link is clicked (on mobile)
document.querySelectorAll('.sidebar-menu a.nav-link').forEach(a=>{
  a.addEventListener('click', ()=>{ if (window.innerWidth <= 991) closeSidebar(); });
});

/* ======= Section switcher ======= */
function showSection(name) {
  document.querySelectorAll('.section').forEach(s => s.classList.remove('active'));
  document.querySelectorAll('.sidebar-menu .nav-link').forEach(a => a.classList.remove('active'));
  const el = document.getElementById(name + '-section');
  if (el) el.classList.add('active');
  const link = document.querySelector('.sidebar-menu a[data-section="'+name+'"]');
  if (link) link.classList.add('active');
  
  // Load transactions when transactions section is opened
  if (name === 'transactions') {
    loadTransactions();
  }
  
  // Initialize Face ID status when Face ID section is opened
  if (name === 'face_id') {
    checkFaceStatus();
  }
}

/* ======= Premium Sidebar Functions ======= */
function togglePremiumSidebar(e) {
  if (e) e.preventDefault();
  const sidebar = document.getElementById('premiumSidebar');
  const overlay = document.getElementById('premiumOverlay');
  const isOpen = sidebar.style.transform === 'translateX(0px)';
  
  if (isOpen) {
    sidebar.style.transform = 'translateX(100%)';
    overlay.style.display = 'none';
  } else {
    sidebar.style.transform = 'translateX(0)';
    overlay.style.display = 'block';
    loadPremiumSidebar();
  }
}

async function loadPremiumSidebar() {
  try {
    // Get current user's subscription and trial info
    const res = await fetch('../api/get_analytics.php');
    const data = await res.json();
    
    const activeSubInfo = document.getElementById('activeSubInfo');
    const trialInfo = document.getElementById('trialInfo');
    const plansContainer = document.getElementById('plansContainer');
    
    // Hide both by default
    activeSubInfo.style.display = 'none';
    trialInfo.style.display = 'none';
    
    // Show active subscription if exists
    if (data.subscription_status === 'premium') {
      activeSubInfo.style.display = 'block';
      document.getElementById('activeSubName').textContent = data.subscription_name;
      document.getElementById('activeSubDays').textContent = `${data.days_remaining} days remaining`;
      document.getElementById('activeSubSearches').textContent = `${data.searches_remaining} searches available`;
    } else if (data.subscription_status === 'trial') {
      trialInfo.style.display = 'block';
      document.getElementById('trialUsed').textContent = `${data.searches_remaining} searches remaining (Free Trial)`;
    }
    
    // Fetch and display all plans
    const plansRes = await fetch('subscribe.php?ajax=1&action=get_plans');
    const plansData = await plansRes.json();
    
    if (plansData.success && plansData.plans && plansData.plans.length > 0) {
      plansContainer.innerHTML = plansData.plans.map(plan => `
        <div style="background: ${plan.card_color}; border: 2px solid ${plan.card_color}; padding: 15px; border-radius: 8px; color: white; cursor: pointer; transition: all 0.3s; box-shadow: 0 4px 12px rgba(0,0,0,0.15);" onmouseover="this.style.transform='translateY(-4px)'; this.style.boxShadow='0 8px 16px rgba(0,0,0,0.25)'" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 12px rgba(0,0,0,0.15)'">
          <div style="font-size: 14px; font-weight: bold; margin-bottom: 5px; opacity: 0.95;">${plan.plan_name}</div>
          <div style="font-size: 20px; font-weight: bold; margin-bottom: 5px;">₹${plan.price}</div>
          <div style="font-size: 12px; opacity: 0.9; margin-bottom: 8px;">
            ${plan.duration_days} days • ${plan.search_limit} searches
          </div>
          <button onclick="purchasePlan(${plan.id})" style="width: 100%; padding: 8px; background: rgba(255,255,255,0.3); border: 1px solid white; color: white; border-radius: 4px; cursor: pointer; font-weight: bold; transition: all 0.2s;" onmouseover="this.style.background='rgba(255,255,255,0.5)'" onmouseout="this.style.background='rgba(255,255,255,0.3)'">
            Buy Now
          </button>
        </div>
      `).join('');
    } else {
      // If no plans loaded, show message
      plansContainer.innerHTML = '<p style="color: white; text-align: center; opacity: 0.8;">Available plans will appear here</p>';
    }
  } catch (error) {
    console.error('Error loading premium sidebar:', error);
    document.getElementById('plansContainer').innerHTML = '<p style="color: white; text-align: center; opacity: 0.8;">⚠️ Error loading plans</p>';
  }
}

function purchasePlan(planId) {
  // Use the same AJAX flow as dashboard cards
  // First, get the plan details
  fetch('subscribe.php?ajax=1&action=get_plans')
    .then(r => r.json())
    .then(data => {
      if (data.success && data.plans) {
        const plan = data.plans.find(p => p.id === planId);
        if (plan) {
          // Use the same function as dashboard
          subscribeToPlanFromDashboard(planId, plan.price.replace(/,/g, ''));
        } else {
          showAlert('danger', 'Plan not found');
        }
      }
    })
    .catch(e => {
      console.error('Error fetching plan:', e);
      showAlert('danger', 'Error loading plan details');
    });
}

/* ======= Real-time Subscription Status Check ======= */
function updateSubscriptionBadge() {
  postForm('check_subscription_status', {})
    .then(result => {
      if (result.success) {
        console.log('Subscription status updated:', result.subscription);
        
        // If subscription is now active, reload to show updated badge
        if (result.subscription && result.subscription.status === 'active') {
          const endDate = new Date(result.subscription.end_date);
          if (endDate > new Date()) {
            // Subscription is active, check if badge needs updating
            const badgeContainer = document.querySelector('.d-flex.align-items-center.gap-3');
            if (badgeContainer) {
              const currentBadge = badgeContainer.innerHTML;
              if (currentBadge.includes('Upgrade Now') || currentBadge.includes('FF3B30')) {
                // Badge shows "Upgrade Now" but subscription is now active - reload
                console.log('Subscription detected! Reloading page...');
                location.reload();
              }
            }
          }
        }
      }
    })
    .catch(e => console.log('Subscription check error (not critical):', e));
}

// Check subscription status less frequently for better performance
setInterval(updateSubscriptionBadge, 30000); // Check every 30 seconds instead of 10
// Don't check immediately on page load to speed up initial load time

/* ======= Alerts ======= */
function showAlert(type, message, timeout=4000) {
  // Also display in the global alert box for fallback
  const alertBox = document.getElementById('global-alert');
  alertBox.innerHTML = `<div class="alert alert-${type} alert-dismissible fade show" role="alert">${message}<button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>`;
  if (timeout>0) setTimeout(()=>{ alertBox.innerHTML = ''; }, timeout);
  
  // Also show in modal popup for better visibility
  const modal = document.getElementById('alertModal');
  const title = document.getElementById('alertModalTitle');
  const body = document.getElementById('alertModalBody');
  const header = document.getElementById('alertModalHeader');
  
  // Set header background based on alert type
  const headerBgMap = {
    'success': '#d4edda',
    'danger': '#f8d7da',
    'warning': '#fff3cd',
    'info': '#d1ecf1',
    'primary': '#d6d8db'
  };
  
  header.style.backgroundColor = headerBgMap[type] || '#f8d7da';
  
  // Set title based on type
  const titleMap = {
    'success': '✅ Success',
    'danger': '❌ Error',
    'warning': '⚠️ Warning',
    'info': 'ℹ️ Information',
    'primary': 'ℹ️ Message'
  };
  
  title.textContent = titleMap[type] || 'Alert';
  body.innerHTML = `<div class="alert alert-${type} mb-0">${message}</div>`;
  
  // Show the modal
  const bootstrapModal = new bootstrap.Modal(modal);
  bootstrapModal.show();
}

/* ======= AJAX helper ======= */
async function postForm(action, data) {
  data = data || {};
  data.action = action;
  const formData = new FormData();
  for (const k in data) formData.append(k, data[k]);
  
  try {
    const resp = await fetch(location.href, { method: 'POST', body: formData });
    
    if (!resp.ok) {
      throw new Error('HTTP ' + resp.status);
    }
    
    const text = await resp.text();
    console.log('postForm response:', text);
    
    const json = JSON.parse(text);
    return json;
  } catch (error) {
    console.error('postForm error:', error);
    return { success: false, message: 'Request failed: ' + error.message };
  }
}

/* ======= Suggestions Dropdown ======= */
async function getSuggestions(type, query) {
  // Suggestions disabled - always hide the dropdown
  document.getElementById(type + '-suggestions').style.display = 'none';
  return;
}

function selectSuggestion(type, value) {
  document.getElementById(type + '-input').value = value;
  document.getElementById(type + '-suggestions').style.display = 'none';
}

document.getElementById('mobile-input').addEventListener('input', function() {
  getSuggestions('mobile', this.value);
});

document.getElementById('mobile-input').addEventListener('blur', function() {
  setTimeout(() => {
    document.getElementById('mobile-suggestions').style.display = 'none';
  }, 200);
});

document.getElementById('mobile-input').addEventListener('focus', function() {
  if (this.value.length > 0) {
    getSuggestions('mobile', this.value);
  }
});

document.getElementById('aadhaar-input').addEventListener('input', function() {
  getSuggestions('aadhaar', this.value);
});

document.getElementById('aadhaar-input').addEventListener('blur', function() {
  setTimeout(() => {
    document.getElementById('aadhaar-suggestions').style.display = 'none';
  }, 200);
});

document.getElementById('aadhaar-input').addEventListener('focus', function() {
  if (this.value.length > 0) {
    getSuggestions('aadhaar', this.value);
  }
});

/* ======= Analytics ======= */
async function loadAnalytics() {
  try {
    const res = await fetch('../api/get_analytics.php');
    const data = await res.json();
    
    if (data.success) {
      // Check if trial user - show loading
      if (data.is_loading) {
        document.getElementById('total-searches').textContent = '...';
        document.getElementById('mobile-searches').textContent = '...';
        document.getElementById('aadhaar-searches').textContent = '...';
        document.getElementById('current-balance').textContent = data.searches_remaining;
        document.getElementById('plan-info').textContent = data.subscription_name;
        document.getElementById('success-rate').textContent = '...';
        document.getElementById('recent-activity').innerHTML = '<p class="text-muted small">Data loading...</p>';
        return;
      }
      
      // Update stats
      document.getElementById('total-searches').textContent = data.total_searches;
      document.getElementById('mobile-searches').textContent = data.mobile_searches;
      document.getElementById('aadhaar-searches').textContent = data.aadhaar_searches;
      
      // Show searches remaining with plan info
      document.getElementById('current-balance').textContent = data.searches_remaining;
      document.getElementById('plan-info').textContent = `${data.subscription_name} (${data.days_remaining > 0 ? data.days_remaining + ' days' : 'Free'})`;
      
      document.getElementById('success-rate').textContent = data.success_rate;
      
      // Update recent activity
      const activityDiv = document.getElementById('recent-activity');
      if (data.recent_searches && data.recent_searches.length > 0) {
        activityDiv.innerHTML = data.recent_searches.map((s, i) => 
          `<div style="padding: 8px 0; border-bottom: 1px solid #eee;">
             <div style="display: flex; justify-content: space-between; align-items: center;">
               <div><span style="font-size: 12px; background: ${s.type === 'mobile' ? '#667eea' : '#4facfe'}; color: white; padding: 2px 8px; border-radius: 4px; margin-right: 8px;">${s.type === 'mobile' ? '📱' : '🆔'}</span>${s.query}</div>
               <small style="color: #999;">${s.time_ago}</small>
             </div>
           </div>`
        ).join('');
      } else {
        activityDiv.innerHTML = '<p class="text-muted small">No searches yet</p>';
      }
    }
  } catch (error) {
    console.error('Analytics error:', error);
    document.getElementById('recent-activity').innerHTML = '<p class="text-danger small">Failed to load analytics</p>';
  }
}

// Load analytics when analytics section is shown
const origShowSection = window.showSection;
window.showSection = function(name) {
  origShowSection(name);
  if (name === 'analytics') {
    loadAnalytics();
  }
};

/* ======= Mobile search ======= */
document.getElementById('mobile-search-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const mobile = document.getElementById('mobile-input').value.trim();
  if (!/^[6-9]\d{9}$/.test(mobile)) { showAlert('warning','Invalid mobile number'); return; }
  const btn = this.querySelector('button[type="submit"]');
  btn.disabled = true; btn.innerText = 'Searching...';
  try {
    const res = await postForm('search_mobile', { mobile });
    if (res.success) {
      renderMobileResult(res.data, res.fallback);
      if (res.balance !== undefined) {
        updateWalletDisplay(res.balance);
      }
      prependRecentSearch('mobile', mobile, res.data?.cost || 0, 'success');
    } else showAlert('danger', res.message || 'Search failed');
  } catch {
    showAlert('danger', 'Request failed');
  } finally {
    btn.disabled = false; btn.innerText = 'Search';
  }
});

/* ======= Aadhaar search ======= */
document.getElementById('aadhaar-search-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const aadhaar = document.getElementById('aadhaar-input').value.trim();
  if (!/^\d{12}$/.test(aadhaar)) { showAlert('warning','Invalid Aadhaar number'); return; }
  const btn = this.querySelector('button[type="submit"]');
  btn.disabled = true; btn.innerText = 'Searching...';
  try {
    const res = await postForm('search_aadhaar', { aadhaar });
    if (res.success) {
      renderAadhaarResult(res.data, res.fallback);
      if (res.balance !== undefined) {
        updateWalletDisplay(res.balance);
      }
      prependRecentSearch('aadhaar', aadhaar, res.data?.cost || 0, 'success');
    } else showAlert('danger', res.message || 'Search failed');
  } catch {
    showAlert('danger', 'Request failed');
  } finally {
    btn.disabled = false; btn.innerText = 'Search';
  }
});

/* ======= Profile update ======= */
document.getElementById('profile-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const name  = document.getElementById('profile-name').value.trim();
  const email = document.getElementById('profile-email').value.trim();
  if (!name || !email) { showAlert('warning','Name and email required'); return; }
  const res = await postForm('update_profile', { name, email });
  if (res.success) {
    document.getElementById('user-name').innerText  = res.name;
    document.getElementById('topbar-name').innerText = res.name;
    showAlert('success', res.message);
  } else showAlert('danger', res.message || 'Failed to update profile');
});

/* ======= Change password (Modal) ======= */
async function submitPasswordChange() {
  const currentPassword = document.getElementById('current-password').value.trim();
  const newPassword = document.getElementById('new-password').value.trim();
  const confirmPassword = document.getElementById('confirm-password').value.trim();
  
  if (!currentPassword || !newPassword || !confirmPassword) { 
    showAlert('warning','All fields are required'); 
    return; 
  }
  
  if (newPassword !== confirmPassword) { 
    showAlert('warning','New passwords do not match'); 
    return; 
  }
  
  if (newPassword.length < 6) { 
    showAlert('warning','Password must be at least 6 characters'); 
    return; 
  }
  
  const btn = document.getElementById('password-submit-btn');
  btn.disabled = true; 
  btn.innerText = 'Updating...';
  
  try {
    const res = await postForm('change_password', { 
      current_password: currentPassword, 
      new_password: newPassword, 
      confirm_password: confirmPassword 
    });
    
    console.log('Password change response:', res);
    
    if (res.success) {
      showAlert('success', res.message || 'Password updated successfully');
      document.getElementById('current-password').value = '';
      document.getElementById('new-password').value = '';
      document.getElementById('confirm-password').value = '';
      
      // Close modal after 1 second
      setTimeout(() => {
        const modal = bootstrap.Modal.getInstance(document.getElementById('passwordModal'));
        if (modal) modal.hide();
      }, 1000);
    } else {
      showAlert('danger', res.message || 'Failed to update password');
    }
  } catch (error) {
    console.error('Password update error:', error);
    showAlert('danger', 'Error: ' + error.message);
  } finally {
    btn.disabled = false; 
    btn.innerText = 'Update Password';
  }
}

/* ======= Add credits ======= */
document.getElementById('add-credits-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const amount = parseFloat(document.getElementById('credit-amount').value);
  if (!amount || amount <= 0) { showAlert('warning','Enter a valid amount'); return; }
  const btn = this.querySelector('button[type="submit"]');
  btn.disabled = true; btn.innerText = 'Adding...';
  const res = await postForm('add_credits', { amount });
  if (res.success) {
    updateWalletDisplay(res.balance);
    showAlert('success', res.message || 'Credits added');
  } else showAlert('danger', res.message || 'Failed to add credits');
  btn.disabled = false; btn.innerText = 'Add Credits';
});

/* ======= Support ticket ======= */
document.getElementById('support-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const title = document.getElementById('support-title').value.trim();
  const body  = document.getElementById('support-body').value.trim();
  if (!title || !body) { showAlert('warning','Please fill title and message'); return; }
  const btn = this.querySelector('button[type="submit"]');
  btn.disabled = true; btn.innerText = 'Submitting...';
  const res = await postForm('submit_ticket', { title, body });
  if (res.success) {
    showAlert('success', res.message || 'Ticket submitted');
    const list = document.getElementById('tickets-list');
    const node = document.createElement('div');
    node.className = 'mb-2';
    node.innerHTML = `<strong>${escapeHtml(title)}</strong><br><small class="text-muted">open • just now</small>`;
    list.prepend(node);
    document.getElementById('support-title').value = '';
    document.getElementById('support-body').value = '';
  } else showAlert('danger', res.message || 'Failed to submit ticket');
  btn.disabled = false; btn.innerText = 'Submit Ticket';
});

/* ======= Render helpers ======= */
function renderMobileResult(data, fallback) {
  const box = document.getElementById('mobile-result');
  let html = '';
  if (Array.isArray(data)) {
    data.forEach(item => { html += mobileCardHtml(item, fallback); });
  } else if (typeof data === 'object' && data) {
    if (data.data && Array.isArray(data.data)) {
      data.data.forEach(item => { html += mobileCardHtml(item, fallback); });
    } else {
      html = mobileCardHtml(data, fallback);
    }
  } else {
    html = `<div class="alert alert-info">No result</div>`;
  }
  box.innerHTML = html;
}
function mobileCardHtml(item, fallback) {
  return `<div class="card search-result-card mb-3 p-3">
    ${fallback
      ? "<div class='alert alert-warning p-2 mb-2'>⚠️</div>"
      : "<div class='alert alert-success p-2 mb-2'>🟢</div>"
    }
    <div><strong>Name:</strong> ${escapeHtml(item.name || '')}</div>
    <div><strong>Mobile:</strong> ${escapeHtml(item.mobile || '')}</div>
    <div><strong>Father's Name:</strong> ${escapeHtml(item.fname || '')}</div>
    <div><strong>Circle:</strong> ${escapeHtml(item.circle || '')}</div>
    <div><strong>Alt No:</strong> ${escapeHtml(item.alt || '')}</div>
    <div><strong>Aadhar Card:</strong> ${escapeHtml(item.id || '')}</div>
    <div><strong>Address:</strong> ${escapeHtml(item.address || '')}</div>
  </div>`;
}

function renderAadhaarResult(data, fallback) {
  const box = document.getElementById('aadhaar-result');
  let html = '';
  if (typeof data === 'object' && data) {
    html += '<div class="card search-result-card p-3">';
    html += (fallback
      ? "<div class='alert alert-warning p-2 mb-2'>⚠️ </div>"
      : "<div class='alert alert-success p-2 mb-2'>🟢 </div>"
    );
    html += `<div><strong>Address:</strong> ${escapeHtml(data.address || '')}</div>`;
    html += `<div><strong>Home District:</strong> ${escapeHtml(data.homeDistName || data.homeDistrict || '')}</div>`;
    html += `<div><strong>Home State:</strong> ${escapeHtml(data.homeStateName || '')}</div>`;
    if (data.memberDetailsList && Array.isArray(data.memberDetailsList)) {
      html += '<hr><h6>Members</h6><ul>';
      data.memberDetailsList.forEach(m => {
        html += `<li>${escapeHtml(m.memberName || '')} — ${escapeHtml(m.releationship_name || '')} (${escapeHtml(m.memberId || '')})</li>`;
      });
      html += '</ul>';
    }
    html += '</div>';
  } else {
    html = '<div class="alert alert-info">No result</div>';
  }
  box.innerHTML = html;
}

function prependRecentSearch(type, query, cost, status) {
  try {
    const tbody = document.querySelector('#recent-searches-table');
    if (!tbody) return;
    const tr = document.createElement('tr');
    tr.innerHTML = `<td><span class="badge bg-${type==='mobile'?'primary':'success'}">${type.toUpperCase()}</span></td>
      <td>${escapeHtml(query)}</td>
      <td>${Number(cost || 0).toFixed(1)}</td>
      <td><span class="badge bg-${status==='success'?'success':'danger'}">${status.toUpperCase()}</span></td>
      <td>${new Date().toLocaleString()}</td>`;
    tbody.prepend(tr);
  } catch (e) {}
}

// Search from Mobile history
async function searchMobileFromHistory(mobile) {
  const input = document.getElementById('mobile-input');
  if (input) input.value = mobile;
  
  const form = document.getElementById('mobile-search-form');
  if (form) {
    const event = new Event('submit', { bubbles: true });
    form.dispatchEvent(event);
  }
}

// Search from Aadhaar history
async function searchAadhaarFromHistory(aadhaar) {
  const input = document.getElementById('aadhaar-input');
  if (input) input.value = aadhaar;
  
  const form = document.getElementById('aadhaar-search-form');
  if (form) {
    const event = new Event('submit', { bubbles: true });
    form.dispatchEvent(event);
  }
}

// Clear search history by type
async function clearSearchType(type) {
  if (!confirm(`Are you sure you want to clear all ${type} searches?`)) {
    return;
  }

  try {
    const response = await fetch(window.location.href, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded'
      },
      body: `action=clear_search_history_by_type&type=${type}`
    });

    const result = await response.json();
    if (result.success) {
      showAlert('success', result.message);
      setTimeout(() => location.reload(), 1000);
    } else {
      showAlert('danger', result.message);
    }
  } catch (error) {
    showAlert('danger', 'Error: ' + error.message);
  }
}

// Clear all search history
async function clearSearchHistory() {
  if (!confirm('Are you sure you want to clear all search history? This cannot be undone.')) {
    return;
  }

  try {
    const response = await fetch(window.location.href, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded'
      },
      body: 'action=clear_search_history'
    });

    const result = await response.json();
    if (result.success) {
      showAlert('success', result.message);
      setTimeout(() => location.reload(), 1000);
    } else {
      showAlert('danger', result.message);
    }
  } catch (error) {
    showAlert('danger', 'Error: ' + error.message);
  }
}

function updateWalletDisplay(amount) {
  document.getElementById('wallet-display').innerText = '₹ ' + Number(amount).toFixed(2);
  document.getElementById('wallet-amount').innerText = Number(amount).toFixed(2);
  const badge = document.getElementById('wallet-badge');
  if (badge) badge.innerText = Number(amount).toFixed(2);
}

function escapeHtml(str) {
  if (!str) return '';
  return String(str)
    .replace(/&/g,'&amp;')
    .replace(/</g,'&lt;')
    .replace(/>/g,'&gt;')
    .replace(/"/g,'&quot;');
}

// Start on dashboard
showSection('dashboard');

/* ======= WALLET PAYMENT FLOW ======= */
// Proceed to payment button click
function proceedToPayment() {
  console.log('Proceed to Payment clicked');
  
  const amount = parseFloat(document.getElementById('wallet-credit-amount').value);
  console.log('Amount:', amount);
  
  if (!amount || amount < 10 || amount > 100000) {
    showAlert('warning', 'Amount must be between ₹10 and ₹100,000');
    return;
  }

  const btn = event.target;
  btn.disabled = true;
  btn.innerText = 'Creating order...';

  const formData = new FormData();
  formData.append('action', 'create_payment_order');
  formData.append('amount', amount);
  
  console.log('Sending fetch to create order...');
  
  fetch('<?php echo $_SERVER['PHP_SELF']; ?>', {
    method: 'POST',
    body: formData
  })
  .then(response => {
    console.log('Response received:', response.status);
    return response.json();
  })
  .then(data => {
    console.log('JSON parsed:', data);
    
    if (data.success) {
      currentOrderId = data.order_id;
      currentAmount = amount;
      
      console.log('✅ Order created:', data.order_id);
      
      // Hide form, show payment info
      document.getElementById('wallet-add-credits-form-container').style.display = 'none';
      document.getElementById('payment-info').style.display = 'block';
      
      // Load payment details
      loadPaymentDetails(amount, data.order_id);
      
      showAlert('success', '✅ Payment details loaded');
    } else {
      showAlert('danger', '❌ Error: ' + (data.message || 'Failed to create order'));
      console.error('Error:', data);
      btn.disabled = false;
      btn.innerText = 'Proceed to Pay';
    }
  })
  .catch(err => {
    console.error('Fetch error:', err);
    showAlert('danger', '❌ Error: ' + err.message);
    btn.disabled = false;
    btn.innerText = 'Proceed to Pay';
  });
}

// Load payment details (QR + UPI)
function loadPaymentDetails(amount, orderId) {
  console.log('Loading payment details for order:', orderId);
  
  fetch('/extract/user/manual_upi_payment.php?action=get_details&order_id=' + orderId)
    .then(r => {
      console.log('Response received, status:', r.status);
      return r.json();
    })
    .then(data => {
      console.log('Parsed JSON:', data);
      
      if (data.success) {
        // Set payment info
        const amountEl = document.getElementById('payment-amount');
        const orderIdEl = document.getElementById('payment-order-id');
        const qrEl = document.getElementById('payment-qr');
        const upiEl = document.getElementById('payment-upi');
        
        console.log('Elements found:', {amountEl, orderIdEl, qrEl, upiEl});
        
        if (amountEl) amountEl.innerText = amount.toFixed(2);
        if (orderIdEl) orderIdEl.innerText = orderId;
        if (qrEl && data.qr_code) {
          qrEl.src = data.qr_code;
          qrEl.style.display = 'block';
        }
        if (upiEl) upiEl.value = data.upi_id || '';
        
        console.log('✅ Payment details loaded successfully');
      } else {
        showAlert('danger', '❌ ' + (data.message || 'Failed to load payment details'));
        console.error('Payment Details Error:', data);
      }
    })
    .catch(err => {
      showAlert('danger', '❌ Error: ' + err.message);
      console.error('Fetch Error:', err);
    });
}

// Copy UPI ID
function copyUPI() {
  const upiInput = document.getElementById('payment-upi');
  upiInput.select();
  document.execCommand('copy');
  showAlert('success', 'UPI ID copied!');
}

// Handle screenshot upload - wrapped in try-catch since it might not exist on page load
try {
  const screenshotInput = document.getElementById('screenshot-input');
  if (screenshotInput) {
    screenshotInput.addEventListener('change', function(e) {
      const file = this.files[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = function(event) {
          document.getElementById('preview-img').src = event.target.result;
          document.getElementById('screenshot-preview').style.display = 'block';
        };
        reader.readAsDataURL(file);
      }
    });
  }
} catch (e) {
  console.log('Screenshot input not found (this is okay)');
}

// Submit payment
function submitPayment() {
  const utr = document.getElementById('payment-utr').value.trim();
  const fileInput = document.getElementById('screenshot-input');
  
  if (!utr) {
    showAlert('warning', 'Please enter Transaction ID/UTR');
    return;
  }
  
  if (!fileInput.files[0]) {
    showAlert('warning', 'Please upload payment screenshot');
    return;
  }
  
  const formData = new FormData();
  formData.append('action', 'submit_manual_payment');
  formData.append('order_id', currentOrderId);
  formData.append('transaction_id', utr);
  formData.append('payment_screenshot', fileInput.files[0]);
  
  const btn = event.target;
  btn.disabled = true;
  btn.innerText = 'Submitting...';
  
  fetch('<?php echo $_SERVER['PHP_SELF']; ?>', {
    method: 'POST',
    body: formData
  })
  .then(r => r.json())
  .then(data => {
    if (data.success) {
      showAlert('success', 'Payment submitted! Admin will verify within 1-2 hours.');
      setTimeout(() => {
        document.getElementById('payment-info').style.display = 'none';
        document.getElementById('wallet-add-credits-form-container').style.display = 'block';
        document.getElementById('wallet-credit-amount').value = '';
        document.getElementById('payment-utr').value = '';
        document.getElementById('screenshot-preview').style.display = 'none';
        document.getElementById('screenshot-input').value = '';
        currentOrderId = null;
        currentAmount = null;
      }, 2000);
    } else {
      showAlert('danger', data.message || 'Failed to submit payment');
    }
    btn.disabled = false;
    btn.innerText = 'Submit Payment';
  })
  .catch(err => {
    showAlert('danger', 'Error: ' + err.message);
    btn.disabled = false;
    btn.innerText = 'Submit Payment';
  });
}

</script>

</body>

<!-- Global Alert/Error Modal -->
<div class="modal fade" id="alertModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header" id="alertModalHeader" style="background-color: #f8d7da; border-bottom: 1px solid #f5c6cb;">
        <h5 class="modal-title" id="alertModalTitle">Alert</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body" id="alertModalBody">
        <!-- Message content will be inserted here -->
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<!-- Password Change Modal -->
<div class="modal fade" id="passwordModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Change Password</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <form id="password-form">
          <div class="mb-3">
            <label class="form-label">Current Password</label>
            <input type="password" id="current-password" class="form-control" required placeholder="Enter your current password">
            <small class="text-muted">We need this to verify it's you</small>
          </div>
          <div class="mb-3">
            <label class="form-label">New Password</label>
            <input type="password" id="new-password" class="form-control" required placeholder="Min 6 characters">
            <small class="text-muted">Use a strong password with mix of characters</small>
          </div>
          <div class="mb-3">
            <label class="form-label">Confirm New Password</label>
            <input type="password" id="confirm-password" class="form-control" required placeholder="Re-enter your new password">
          </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-warning" id="password-submit-btn" onclick="submitPasswordChange()">Update Password</button>
      </div>
    </div>
  </div>
</div>

<!-- AI Chat Widget -->
<div id="ai-chat-widget" style="position: fixed; bottom: 20px; right: 20px; width: 380px; max-width: 95vw; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif; z-index: 10000; display: none; flex-direction: column; border-radius: 12px; box-shadow: 0 5px 40px rgba(0,0,0,0.16); background: white; height: 550px;">
  <!-- Header -->
  <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 16px; border-radius: 12px 12px 0 0; display: flex; justify-content: space-between; align-items: center;">
    <div>
      <h6 style="margin: 0; font-weight: 600; font-size: 14px;" id="ai-header-title">🤖 AI Assistant</h6>
      <small style="opacity: 0.9; font-size: 12px;" id="ai-header-subtitle">Quick answers in seconds</small>
    </div>
    <div style="display: flex; gap: 8px; align-items: center;">
      <button id="ai-back-btn" onclick="switchToAIChat()" style="display: none; background: none; border: none; color: white; font-size: 16px; cursor: pointer; padding: 4px 8px;">← Back</button>
      <button onclick="toggleAIChat()" style="background: none; border: none; color: white; font-size: 20px; cursor: pointer; padding: 0; width: 24px; height: 24px; display: flex; align-items: center; justify-content: center;">×</button>
    </div>
  </div>
  
  <!-- AI Chat View -->
  <div id="ai-chat-view" style="flex: 1; display: flex; flex-direction: column; gap: 0; background: #f8f9fa; overflow: hidden;">
    <!-- Quick buttons (top, scrollable if many messages) -->
    <div style="padding: 12px; background: white; border-bottom: 1px solid #e5e5e5;">
      <small style="color: #999; display: block; font-size: 12px; margin-bottom: 8px; text-align: center;">👋 Quick answers:</small>
      <div style="display: flex; flex-wrap: wrap; gap: 6px; justify-content: center; margin-bottom: 8px;">
        <button onclick="sendQuickMessage('💰 How many credits?')" style="background: #667eea; color: white; border: none; border-radius: 16px; padding: 6px 12px; font-size: 11px; cursor: pointer; transition: all 0.2s;" onmouseover="this.style.background='#5568d3'" onmouseout="this.style.background='#667eea'">💰 Credits</button>
        <button onclick="sendQuickMessage('🔍 How to search?')" style="background: #764ba2; color: white; border: none; border-radius: 16px; padding: 6px 12px; font-size: 11px; cursor: pointer; transition: all 0.2s;" onmouseover="this.style.background='#65408f'" onmouseout="this.style.background='#764ba2'">🔍 Search</button>
        <button onclick="sendQuickMessage('💳 Payment help')" style="background: #f093fb; color: white; border: none; border-radius: 16px; padding: 6px 12px; font-size: 11px; cursor: pointer; transition: all 0.2s;" onmouseover="this.style.background='#e07ae5'" onmouseout="this.style.background='#f093fb'">💳 Payment</button>
        <button onclick="sendQuickMessage('📊 My statistics')" style="background: #4facfe; color: white; border: none; border-radius: 16px; padding: 6px 12px; font-size: 11px; cursor: pointer; transition: all 0.2s;" onmouseover="this.style.background='#3498db'" onmouseout="this.style.background='#4facfe'">📊 Stats</button>
      </div>
      <button onclick="switchToAdminChat()" style="background: linear-gradient(135deg, #f5576c 0%, #f093fb 100%); color: white; border: none; border-radius: 8px; padding: 8px 16px; font-size: 12px; cursor: pointer; width: 100%; font-weight: 600; transition: all 0.2s;" onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 12px rgba(245,87,108,0.3)'" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
        💬 Contact Admin
      </button>
    </div>
    <!-- Messages container (scrollable - main area) -->
    <div id="ai-chat-messages" style="flex: 1; overflow-y: auto; padding: 12px; display: flex; flex-direction: column; gap: 8px;"></div>
  </div>
  
  <!-- Admin Chat View (Hidden by default) -->
  <div id="admin-chat-view" style="flex: 1; display: none; flex-direction: column; gap: 8px; background: #f8f9fa; overflow: hidden;">
    <div id="admin-chat-messages" style="flex: 1; overflow-y: auto; display: flex; flex-direction: column; gap: 8px; padding: 12px;"></div>
  </div>
  
  <!-- Input Area -->
  <div style="border-top: 1px solid #eee; padding: 10px; display: flex; gap: 8px; background: white;">
    <input type="text" id="ai-chat-input" placeholder="Type your question..." style="flex: 1; border: 1px solid #ddd; border-radius: 20px; padding: 8px 14px; font-size: 13px; outline: none; transition: border 0.2s;" onfocus="this.style.borderColor='#667eea'" onblur="this.style.borderColor='#ddd'" onkeypress="if(event.key==='Enter') sendChatMessage()">
    <button onclick="sendChatMessage()" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 50%; width: 36px; height: 36px; cursor: pointer; display: flex; align-items: center; justify-content: center; font-size: 18px; transition: transform 0.2s;" onmouseover="this.style.transform='scale(1.1)'" onmouseout="this.style.transform='scale(1)'">→</button>
  </div>
</div>

<!-- Chat Bubble -->
<button id="ai-chat-bubble" onclick="toggleAIChat()" style="position: fixed; bottom: 20px; right: 20px; width: 56px; height: 56px; border-radius: 50%; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; cursor: pointer; font-size: 24px; display: flex; align-items: center; justify-content: center; box-shadow: 0 2px 12px rgba(102, 126, 234, 0.4); z-index: 9999; transition: transform 0.2s;" onmouseover="this.style.transform='scale(1.1)'" onmouseout="this.style.transform='scale(1)'">🤖</button>

<style>
#ai-chat-widget .message-user {
  background: #667eea;
  color: white;
  padding: 10px 14px;
  border-radius: 12px;
  margin-left: auto;
  max-width: 85%;
  word-wrap: break-word;
  font-size: 13px;
  line-height: 1.4;
  box-shadow: 0 1px 3px rgba(0,0,0,0.1);
}

#ai-chat-widget .message-ai {
  background: #e8e8f0;
  color: #333;
  padding: 10px 14px;
  border-radius: 12px;
  margin-right: auto;
  max-width: 85%;
  word-wrap: break-word;
  font-size: 13px;
  line-height: 1.5;
  box-shadow: 0 1px 3px rgba(0,0,0,0.05);
}

#ai-chat-widget .message-ai strong {
  color: #667eea;
  font-weight: 600;
}

#ai-chat-widget .suggestion-button {
  background: #f0f4ff;
  color: #667eea;
  border: 1px solid #667eea;
  border-radius: 8px;
  padding: 8px 12px;
  margin: 4px;
  font-size: 12px;
  cursor: pointer;
  transition: all 0.2s;
}

#ai-chat-widget .suggestion-button:hover {
  background: #667eea;
  color: white;
  transform: translateY(-2px);
}
</style>

<script>
// Global state for admin chat
let adminChatTicketId = null;
let adminChatLastMessageId = 0;
let adminChatPollInterval = null;

function toggleAIChat() {
  const widget = document.getElementById('ai-chat-widget');
  const bubble = document.getElementById('ai-chat-bubble');
  if (widget.style.display === 'flex') {
    widget.style.display = 'none';
    bubble.style.display = 'flex';
  } else {
    widget.style.display = 'flex';
    bubble.style.display = 'none';
    document.getElementById('ai-chat-input').focus();
  }
}

function switchToAdminChat() {
  // Hide AI view, show admin view
  document.getElementById('ai-chat-view').style.display = 'none';
  document.getElementById('admin-chat-view').style.display = 'flex';
  document.getElementById('ai-header-title').textContent = '💬 Admin Chat';
  document.getElementById('ai-header-subtitle').textContent = 'Chat with our support team';
  document.getElementById('ai-back-btn').style.display = 'block';
  document.getElementById('ai-chat-input').placeholder = 'Type your message...';
  
  // Initialize admin chat if first time
  if (!adminChatTicketId) {
    initializeAdminChat();
  }
}

function switchToAIChat() {
  // Hide admin view, show AI view
  document.getElementById('ai-chat-view').style.display = 'flex';
  document.getElementById('admin-chat-view').style.display = 'none';
  document.getElementById('ai-header-title').textContent = '🤖 AI Assistant';
  document.getElementById('ai-header-subtitle').textContent = 'Quick answers in seconds';
  document.getElementById('ai-back-btn').style.display = 'none';
  document.getElementById('ai-chat-input').placeholder = 'Type your question...';
  
  // Stop polling when switching away
  if (adminChatPollInterval) {
    clearInterval(adminChatPollInterval);
    adminChatPollInterval = null;
  }
}

async function initializeAdminChat() {
  try {
    const res = await fetch('../api/admin_chat.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: 'action=get_or_create_ticket'
    });
    
    const data = await res.json();
    if (data.success) {
      adminChatTicketId = data.ticket_id;
      adminChatLastMessageId = 0;
      
      // Load existing messages
      await loadAdminChatMessages();
      
      // Start polling for new admin messages
      startAdminChatPolling();
    }
  } catch (err) {
    console.error('Failed to initialize admin chat:', err);
  }
}

async function loadAdminChatMessages() {
  if (!adminChatTicketId) return;
  
  try {
    const res = await fetch('../api/admin_chat.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: 'action=get_messages&ticket_id=' + adminChatTicketId
    });
    
    const data = await res.json();
    if (data.success) {
      const messagesDiv = document.getElementById('admin-chat-messages');
      messagesDiv.innerHTML = '';
      
      data.messages.forEach(msg => {
        appendAdminChatMessage(msg);
        adminChatLastMessageId = Math.max(adminChatLastMessageId, msg.id);
      });
      
      messagesDiv.scrollTop = messagesDiv.scrollHeight;
    }
  } catch (err) {
    console.error('Failed to load admin chat messages:', err);
  }
}

function appendAdminChatMessage(msg) {
  const messagesDiv = document.getElementById('admin-chat-messages');
  const msgEl = document.createElement('div');
  msgEl.style.cssText = msg.sender_type === 'admin'
    ? 'background: #e8f0fe; color: #333; padding: 10px 14px; border-radius: 12px; margin-left: auto; max-width: 80%;'
    : 'background: #667eea; color: white; padding: 10px 14px; border-radius: 12px; margin-right: auto; max-width: 80%;';
  
  const timeStr = new Date(msg.created_at).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
  msgEl.innerHTML = `<small style="opacity: 0.7; font-size: 10px;">${msg.sender_type === 'admin' ? '👨‍💼 Admin' : 'You'} ${timeStr}</small><br>${msg.message}`;
  
  messagesDiv.appendChild(msgEl);
}

function startAdminChatPolling() {
  // Poll every 2 seconds for new admin messages
  if (adminChatPollInterval) clearInterval(adminChatPollInterval);
  
  adminChatPollInterval = setInterval(async () => {
    if (!adminChatTicketId) return;
    
    try {
      const res = await fetch('../api/admin_chat.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'action=poll_messages&ticket_id=' + adminChatTicketId + '&last_message_id=' + adminChatLastMessageId
      });
      
      const data = await res.json();
      if (data.success && data.messages && data.messages.length > 0) {
        data.messages.forEach(msg => {
          appendAdminChatMessage(msg);
          adminChatLastMessageId = Math.max(adminChatLastMessageId, msg.id);
        });
        
        const messagesDiv = document.getElementById('admin-chat-messages');
        messagesDiv.scrollTop = messagesDiv.scrollHeight;
      }
    } catch (err) {
      console.error('Polling error:', err);
    }
  }, 5000); // Poll every 5 seconds instead of 2 for better performance
}

async function sendChatMessage() {
  const input = document.getElementById('ai-chat-input');
  const message = input.value.trim();
  if (!message) return;
  
  // Determine if we're in AI or Admin chat
  const isAdminChat = document.getElementById('admin-chat-view').style.display !== 'none';
  
  if (isAdminChat) {
    // Send to admin chat
    if (!adminChatTicketId) return;
    
    try {
      const res = await fetch('../api/admin_chat.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'action=send_message&ticket_id=' + adminChatTicketId + '&message=' + encodeURIComponent(message)
      });
      
      const data = await res.json();
      if (data.success) {
        appendAdminChatMessage(data.message);
        input.value = '';
        
        const messagesDiv = document.getElementById('admin-chat-messages');
        messagesDiv.scrollTop = messagesDiv.scrollHeight;
      }
    } catch (err) {
      console.error('Failed to send message:', err);
    }
  } else {
    // Send to AI chat
    sendAIMessage();
  }
}

function sendQuickMessage(message) {
  document.getElementById('ai-chat-input').value = message;
  sendAIMessage();
}

async function sendAIMessage() {
  const input = document.getElementById('ai-chat-input');
  const message = input.value.trim();
  if (!message) return;
  
  const messagesDiv = document.getElementById('ai-chat-messages');
  
  // Clear welcome message if this is first message
  if (messagesDiv.children.length === 0) {
    messagesDiv.innerHTML = '';
  }
  
  // Add user message
  const userMsg = document.createElement('div');
  userMsg.className = 'message-user';
  userMsg.textContent = message;
  userMsg.style.marginBottom = '4px';
  messagesDiv.appendChild(userMsg);
  
  input.value = '';
  messagesDiv.scrollTop = messagesDiv.scrollHeight;
  
  // Add typing indicator
  const typingDiv = document.createElement('div');
  typingDiv.id = 'typing-indicator';
  typingDiv.style.cssText = 'margin-right: auto; color: #999; font-size: 12px; font-style: italic;';
  typingDiv.textContent = '⏳ AI is thinking...';
  messagesDiv.appendChild(typingDiv);
  messagesDiv.scrollTop = messagesDiv.scrollHeight;
  
  try {
    const res = await fetch('../api/ai_chat.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: 'message=' + encodeURIComponent(message)
    });
    
    const data = await res.json();
    
    // Remove typing indicator
    if (typingDiv.parentNode) typingDiv.parentNode.removeChild(typingDiv);
    
    // Add AI response with clickable suggestions
    const aiMsg = document.createElement('div');
    aiMsg.className = 'message-ai';
    aiMsg.innerHTML = data.response
      .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
      .replace(/\n/g, '<br>');
    aiMsg.style.marginBottom = '4px';
    
    // Add click handlers to text for quick follow-ups
    aiMsg.addEventListener('click', function(e) {
      if (e.target.tagName !== 'BR') {
        const text = e.target.textContent || e.target.innerText;
        if (text && text.length > 3 && text.length < 100) {
          document.getElementById('ai-chat-input').value = 'Tell me more about: ' + text;
          document.getElementById('ai-chat-input').focus();
        }
      }
    });
    aiMsg.style.cursor = 'pointer';
    
    messagesDiv.appendChild(aiMsg);
    messagesDiv.scrollTop = messagesDiv.scrollHeight;
  } catch (error) {
    console.error('Chat error:', error);
    
    // Remove typing indicator
    if (typingDiv.parentNode) typingDiv.parentNode.removeChild(typingDiv);
    
    const errorMsg = document.createElement('div');
    errorMsg.className = 'message-ai';
    errorMsg.textContent = '❌ Oops! Something went wrong. Please try again.';
    errorMsg.style.marginBottom = '4px';
    messagesDiv.appendChild(errorMsg);
  }
}

/* ======= FACE ID FUNCTIONS ======= */
let faceStream = null;

function checkFaceStatus() {
  fetch('../api/face_recognition.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: 'action=check_face_enabled'
  })
  .then(res => res.json())
  .then(data => {
    const statusDiv = document.getElementById('face-status');
    const statusText = document.getElementById('face-status-text');
    const regForm = document.getElementById('face-registration-form');
    const delForm = document.getElementById('face-delete-form');
    
    if (data.has_face) {
      statusText.innerHTML = '✓ Face is registered and ready for login!';
      statusDiv.className = 'alert alert-success';
      regForm.style.display = 'none';
      delForm.style.display = 'block';
    } else {
      statusText.innerHTML = '⚠ No face registered yet. Register below to enable Face ID login.';
      statusDiv.className = 'alert alert-warning';
      regForm.style.display = 'block';
      delForm.style.display = 'none';
    }
  })
  .catch(err => {
    console.error('Face status check failed:', err);
    document.getElementById('face-status-text').textContent = 'Error checking status';
  });
}

function startFaceCamera() {
  const video = document.getElementById('face-video');
  const captureBtn = document.getElementById('capture-face-btn');
  
  if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
    navigator.mediaDevices.getUserMedia({
      video: {
        width: { ideal: 640 },
        height: { ideal: 480 },
        facingMode: 'user'
      },
      audio: false
    })
    .then(stream => {
      faceStream = stream;
      video.srcObject = stream;
      captureBtn.style.display = 'block';
      
      const msgDiv = document.getElementById('face-messages');
      msgDiv.innerHTML = '<div class="alert alert-info"><i data-feather="camera"></i> ✓ Camera started. Position your face clearly and click "Capture Face".</div>';
      feather.replace();
    })
    .catch(err => {
      console.error('Camera access error:', err);
      const msgDiv = document.getElementById('face-messages');
      let errorMsg = '';
      
      // Provide specific error messages
      if (err.name === 'NotAllowedError' || err.name === 'PermissionDeniedError') {
        errorMsg = '<div class="alert alert-warning"><strong>❌ Camera Access Denied:</strong><br>' +
          'You denied camera access. To fix this:<br>' +
          '1. Click the camera/lock icon in your browser address bar<br>' +
          '2. Find "Camera" permission<br>' +
          '3. Select "Allow" instead of "Block"<br>' +
          '4. Reload this page and try again</div>';
      } else if (err.name === 'NotFoundError' || err.name === 'DevicesNotFoundError') {
        errorMsg = '<div class="alert alert-danger"><strong>❌ No Camera Found:</strong><br>Your device does not have a camera, or it is not connected.</div>';
      } else if (err.name === 'NotSupportedError') {
        errorMsg = '<div class="alert alert-danger"><strong>❌ HTTPS Required:</strong><br>Please access this page over HTTPS for camera access.</div>';
      } else {
        errorMsg = '<div class="alert alert-danger"><strong>❌ Camera Error:</strong> ' + err.name + ': ' + err.message + '</div>';
      }
      
      msgDiv.innerHTML = errorMsg;
    });
  } else {
    alert('Your browser does not support camera access. Please use a modern browser (Chrome, Firefox, Safari, Edge).');
  }
}

function stopFaceCamera() {
  if (faceStream) {
    faceStream.getTracks().forEach(track => track.stop());
    faceStream = null;
  }
  document.getElementById('face-video').srcObject = null;
  document.getElementById('capture-face-btn').style.display = 'none';
  document.getElementById('face-messages').innerHTML = '';
}

function captureFace() {
  const video = document.getElementById('face-video');
  const canvas = document.getElementById('face-canvas');
  const ctx = canvas.getContext('2d');
  
  // Set canvas size to match video
  canvas.width = video.videoWidth;
  canvas.height = video.videoHeight;
  
  // Draw video frame to canvas
  ctx.drawImage(video, 0, 0);
  
  // Get face detection using tracking.js
  const tracker = new tracking.ObjectTracker('face');
  tracker.setInitialScale(4);
  tracker.setStepSize(2);
  tracker.setEdgesDensity(0.1);
  
  tracking.track('#face-canvas', tracker);
  
  tracker.on('track', function(event) {
    if (event.data.length === 0) {
      console.error('No face detected');
      const msgDiv = document.getElementById('face-messages');
      msgDiv.innerHTML = '<div class="alert alert-warning">❌ No face detected. Please try again with better lighting and a clear view.</div>';
      return;
    }
    
    // Face detected, extract pixel data as descriptor
    const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
    const pixelData = imageData.data;
    
    // Create descriptor from first 2000 bytes of pixel data
    const descriptor = btoa(String.fromCharCode.apply(null, Array.from(pixelData.slice(0, 2000))));
    
    console.log('Face captured, descriptor length:', descriptor.length);
    
    // Send to API
    const formData = new FormData();
    formData.append('action', 'register_face');
    formData.append('face_data', descriptor);
    
    fetch('../api/face_recognition.php', {
      method: 'POST',
      body: formData
    })
    .then(res => res.json())
    .then(data => {
      console.log('API Response:', data);
      if (data.success) {
        const msgDiv = document.getElementById('face-messages');
        msgDiv.innerHTML = '<div class="alert alert-success"><strong>✓ Face registered successfully!</strong> You can now use Face ID to log in.</div>';
        
        // Stop camera
        stopFaceCamera();
        
        // Refresh status
        setTimeout(checkFaceStatus, 1000);
      } else {
        const msgDiv = document.getElementById('face-messages');
        msgDiv.innerHTML = '<div class="alert alert-danger"><strong>Error:</strong> ' + (data.error || 'Failed to register face') + '</div>';
        console.error('Registration failed:', data);
      }
    })
    .catch(err => {
      console.error('API error:', err);
      const msgDiv = document.getElementById('face-messages');
      msgDiv.innerHTML = '<div class="alert alert-danger"><strong>Error:</strong> ' + err.message + '</div>';
    });
  });
}

function deleteFaceData() {
  if (!confirm('Are you sure you want to delete your face registration? You can register again anytime.')) {
    return;
  }
  
  fetch('../api/face_recognition.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: 'action=delete_face'
  })
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      showAlert('success', '✓ Face registration deleted successfully');
      setTimeout(checkFaceStatus, 500);
    } else {
      showAlert('danger', '❌ Error deleting face: ' + (data.error || 'Unknown error'));
    }
  })
  .catch(err => {
    console.error('Delete error:', err);
    showAlert('danger', '❌ Error: ' + err.message);
  });
}

// Initialize Face ID section when it's shown
const originalShowSection = showSection;
window.showSection = function(name) {
  originalShowSection(name);
  if (name === 'face_id') {
    // Ensure tracking.js is loaded
    if (typeof tracking === 'undefined') {
      console.log('Loading tracking.js...');
      const script = document.createElement('script');
      script.src = 'https://cdnjs.cloudflare.com/ajax/libs/tracking.js/0.7.3/tracking.min.js';
      script.onload = () => {
        console.log('tracking.js loaded');
        // Load face detection model
        const faceScript = document.createElement('script');
        faceScript.src = 'https://cdnjs.cloudflare.com/ajax/libs/tracking.js/0.7.3/data/face.min.js';
        faceScript.onload = () => {
          console.log('Face model loaded');
          checkFaceStatus();
        };
        document.head.appendChild(faceScript);
      };
      script.onerror = () => {
        console.error('Failed to load tracking.js');
        const msgDiv = document.getElementById('face-messages');
        msgDiv.innerHTML = '<div class="alert alert-danger">Failed to load face detection library. Please refresh the page and try again.</div>';
      };
      document.head.appendChild(script);
    } else {
      console.log('tracking.js already loaded');
      checkFaceStatus();
    }
  }
};

// ======== Load Transactions on Section Show ========
document.addEventListener('DOMContentLoaded', function() {
  // Listen for section changes with debouncing for performance
  let loadTimeout;
  const observer = new MutationObserver(function(mutations) {
    clearTimeout(loadTimeout);
    loadTimeout = setTimeout(() => {
      const transSection = document.getElementById('transactions-section');
      if (transSection && transSection.classList.contains('active')) {
        loadTransactions();
      }
    }, 300); // Debounce to avoid rapid repeated calls
  });
  
  const sidebar = document.querySelector('.main-content');
  if (sidebar) {
    observer.observe(sidebar, { attributes: true, attributeFilter: ['class'] }); // Only observe class changes
  }
  
  // Also check on page load
  setTimeout(() => {
    const transSection = document.getElementById('transactions-section');
    if (transSection && transSection.classList.contains('active')) {
      loadTransactions();
    }
  }, 500);
});

function loadTransactions() {
  console.log('Loading transactions...');
  
  // Load available plans
  fetch('subscribe.php?ajax=1&action=get_plans')
    .then(r => r.json())
    .then(data => {
      const container = document.getElementById('available-plans-container');
      if (container) {
        if (data.success && data.plans && data.plans.length > 0) {
          container.innerHTML = data.plans.map(plan => `
            <div class="col-md-6 col-lg-4">
              <div class="card h-100" style="border: 2px solid ${plan.card_color}; background: linear-gradient(135deg, ${plan.card_color}15 0%, ${plan.card_color}08 100%); transition: all 0.3s ease;">
                <div class="card-body" style="padding: 20px;">
                  <div style="background: ${plan.card_color}; color: white; padding: 12px 16px; border-radius: 8px; margin-bottom: 15px; text-align: center;">
                    <h5 style="margin: 0; font-weight: bold; font-size: 18px;">${plan.plan_name}</h5>
                  </div>
                  
                  <div style="text-align: center; margin-bottom: 15px;">
                    <div style="font-size: 28px; font-weight: bold; color: ${plan.card_color};">₹${plan.price}</div>
                    <small style="color: #666;">for ${plan.duration_days} days</small>
                  </div>
                  
                  <div style="background: white; padding: 12px; border-radius: 6px; margin-bottom: 15px; text-align: center;">
                    <div style="font-size: 24px; font-weight: bold; color: #333;">${plan.search_limit}</div>
                    <small style="color: #666;">Searches</small>
                  </div>
                  
                  <button class="btn w-100" style="background: ${plan.card_color}; color: white; border: none; padding: 10px; border-radius: 6px; font-weight: bold; transition: all 0.2s; cursor: pointer;" onmouseover="this.style.opacity='0.9'; this.style.transform='translateY(-2px)'" onmouseout="this.style.opacity='1'; this.style.transform='translateY(0)'" onclick="subscribeToPlanFromDashboard(${plan.id}, '${plan.price.replace(/,/g, '')}'); return false;">
                    Subscribe Now
                  </button>
                </div>
              </div>
            </div>
          `).join('');
        } else {
          container.innerHTML = '<div class="alert alert-info w-100">No plans available at the moment</div>';
        }
      }
      
      // Re-apply feather icons if they exist
      if (typeof feather !== 'undefined') {
        feather.replace();
      }
      console.log('Plans loaded successfully');
    })
    .catch(e => {
      console.error('Failed to load plans:', e);
      const container = document.getElementById('available-plans-container');
      if (container) container.innerHTML = '<div class="alert alert-danger">Failed to load plans</div>';
    });
  
  // Load purchased plans/subscriptions
  console.log('Fetching user subscriptions...');
  postForm('get_my_transactions', {})
    .then(result => {
      console.log('Transactions result:', result);
      if (result.success && result.transactions) {
        const tbody = document.getElementById('purchased-plans-tbody');
        if (tbody) {
          if (result.transactions.length === 0) {
            tbody.innerHTML = '<tr><td colspan="8" class="text-muted text-center">No purchases yet</td></tr>';
          } else {
            tbody.innerHTML = result.transactions.map(t => {
              const endDate = new Date(t.end_date);
              const statusColor = t.status === 'active' && endDate > new Date() ? '#34C759' : '#FF9500';
              const statusText = t.status === 'active' && endDate > new Date() ? 'Active' : 'Expired';
              const orderId = t.razorpay_order_id ? t.razorpay_order_id.substring(0, 10) + '...' : 'N/A';
              const paymentId = t.razorpay_payment_id ? t.razorpay_payment_id.substring(0, 10) + '...' : 'N/A';
              const isExpired = endDate <= new Date();
              const actionBtn = isExpired 
                ? `<button class="btn btn-sm btn-success" style="padding: 4px 8px; font-size: 11px;" onclick="showPlansModal()">Renew</button>`
                : '<span style="color: #999; font-size: 11px;">Active</span>';
              
              return `
                <tr style="font-size: 0.9rem;">
                  <td><strong>${t.plan_name}</strong></td>
                  <td>${t.duration_days} days</td>
                  <td>₹${parseFloat(t.price).toFixed(0)}</td>
                  <td title="${t.razorpay_order_id || 'Not available'}" style="cursor: help; color: #666;">${orderId}</td>
                  <td title="${t.razorpay_payment_id || 'Not available'}" style="cursor: help; color: #666;">${paymentId}</td>
                  <td>
                    <span style="background: ${statusColor}; color: white; padding: 4px 8px; border-radius: 4px; font-size: 11px;">
                      ${statusText}
                    </span>
                  </td>
                  <td>${endDate.toLocaleDateString()}</td>
                  <td>${actionBtn}</td>
                </tr>
              `;
            }).join('');
          }
        }
      } else {
        const tbody = document.getElementById('purchased-plans-tbody');
        if (tbody) tbody.innerHTML = '<tr><td colspan="8" class="text-muted text-center">No purchases yet</td></tr>';
      }
    })
    .catch(e => {
      console.error('Failed to load transactions:', e);
      const tbody = document.getElementById('purchased-plans-tbody');
      if (tbody) tbody.innerHTML = '<tr><td colspan="8" class="text-danger text-center">Error loading subscriptions</td></tr>';
    });
}

// Show plans section for renewal or new purchase
function showPlansModal() {
  // Scroll to the available plans section
  const plansContainer = document.getElementById('available-plans-container');
  if (plansContainer) {
    plansContainer.scrollIntoView({ behavior: 'smooth' });
    // Optionally highlight it
    plansContainer.style.backgroundColor = '#ffffcc';
    setTimeout(() => {
      plansContainer.style.backgroundColor = '';
    }, 2000);
  }
}

// Purchase plan function for dashboard
async function subscribeToPlanFromDashboard(planId, amount) {
  // Step 1: Create payment order
  try {
    console.log('Starting subscription for plan:', planId, 'amount:', amount);
    
    const formData = new URLSearchParams();
    formData.append('action', 'create_payment_order');
    formData.append('plan_id', planId);
    
    const response = await fetch('subscribe.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded'
      },
      body: formData.toString()
    });
    
    console.log('Response status:', response.status);
    const text = await response.text();
    console.log('Raw response:', text);
    
    let result;
    try {
      result = JSON.parse(text);
    } catch (e) {
      console.error('Failed to parse response as JSON:', e);
      showAlert('danger', 'Server error: Invalid response');
      return;
    }
    
    if (result.success) {
      console.log('Order created successfully:', result);
      openPaymentWindow(result);
    } else {
      console.error('Order creation failed:', result);
      showAlert('danger', 'Error: ' + (result.message || 'Unknown error'));
    }
  } catch (error) {
    console.error('Error creating payment order:', error);
    showAlert('danger', 'Error creating payment order: ' + error.message);
  }
}

// Open Razorpay payment window
function openPaymentWindow(orderData) {
  if (typeof Razorpay === 'undefined') {
    showAlert('danger', 'Payment gateway not loaded. Please refresh the page.');
    return;
  }
  
  var options = {
    key: orderData.key_id,
    amount: orderData.amount * 100,
    currency: 'INR',
    name: 'Advanced Mobile Info System',
    description: orderData.plan_name,
    order_id: orderData.order_id,
    handler: function(response) {
      verifyPaymentFromDashboard(response, orderData.subscription_id);
    },
    prefill: {
      email: document.querySelector('.sidebar-header small')?.textContent || '',
      contact: ''
    },
    theme: {
      color: '#007AFF'
    }
  };
  
  var rzp = new Razorpay(options);
  rzp.open();
}

// Verify payment
async function verifyPaymentFromDashboard(paymentResponse, subscriptionId) {
  try {
    console.log('Verifying payment for subscription:', subscriptionId);
    
    const formData = new URLSearchParams();
    formData.append('action', 'verify_payment');
    formData.append('subscription_id', subscriptionId);
    formData.append('order_id', paymentResponse.razorpay_order_id);
    formData.append('payment_id', paymentResponse.razorpay_payment_id);
    formData.append('signature', paymentResponse.razorpay_signature);
    
    const response = await fetch('subscribe.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded'
      },
      body: formData.toString()
    });
    
    console.log('Verification response status:', response.status);
    const text = await response.text();
    console.log('Raw verification response:', text);
    
    let result;
    try {
      result = JSON.parse(text);
    } catch (e) {
      console.error('Failed to parse verification response:', e);
      showAlert('danger', 'Server error: Invalid response');
      return;
    }
    
    if (result.success) {
      console.log('Payment verified successfully');
      showAlert('success', '✓ ' + result.message);
      // Refresh transactions after 1.5 seconds
      setTimeout(() => {
        loadTransactions();
      }, 1500);
      // Full page reload after 3 seconds
      setTimeout(() => {
        location.reload();
      }, 3000);
    } else {
      console.error('Payment verification failed:', result);
      showAlert('danger', 'Payment verification failed: ' + (result.message || 'Unknown error'));
    }
  } catch (error) {
    console.error('Error verifying payment:', error);
    showAlert('danger', 'Error verifying payment: ' + error.message);
  }
}


</script>

</html>
