<?php
/**
 * Face ID Registration for Users
 * Allows users to register/manage their face for login
 */

require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';
require_once '../classes/FaceRecognition.php';

$db = Database::getInstance();
$auth = new Auth($db);
$faceRecognition = new FaceRecognition($db);

if (!$auth->isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

$currentUser = $auth->getCurrentUser();
$hasFaceData = $faceRecognition->hasFaceData($currentUser['id']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Face ID Settings - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://unpkg.com/feather-icons"></script>
    <!-- Tracking.js for face detection -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/tracking.js/1.9.3/tracking.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/tracking.js/1.9.3/data/face.min.js"></script>
    
    <style>
        body {
            background: #f5f7fa;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }
        
        .container {
            max-width: 600px;
            margin-top: 2rem;
        }
        
        .card {
            border: none;
            border-radius: 16px;
            box-shadow: 0 4px 18px rgba(0,0,0,0.1);
        }
        
        .card-header {
            background: linear-gradient(135deg, #007AFF 0%, #0056CC 100%);
            color: white;
            border-radius: 16px 16px 0 0 !important;
            padding: 1.5rem;
        }
        
        #face-video {
            width: 100%;
            max-width: 300px;
            border-radius: 12px;
            background: #000;
            margin: 1rem auto;
            display: block;
        }
        
        .btn-success {
            background: linear-gradient(135deg, #34C759 0%, #00B741 100%);
            border: none;
        }
        
        .btn-success:hover {
            background: linear-gradient(135deg, #00B741 0%, #008030 100%);
        }
        
        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-weight: 600;
            display: inline-block;
            margin: 0.5rem 0;
        }
        
        .status-badge.active {
            background: rgba(52, 199, 89, 0.2);
            color: #00B741;
        }
        
        .status-badge.inactive {
            background: rgba(142, 142, 147, 0.2);
            color: #8E8E93;
        }
        
        .info-box {
            background: rgba(0, 122, 255, 0.1);
            border-left: 4px solid #007AFF;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1rem;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <div class="card-header">
                <h4 class="mb-0">
                    <i data-feather="camera" class="me-2"></i> Face ID Settings
                </h4>
            </div>
            
            <div class="card-body p-4">
                <!-- Current Status -->
                <h5 class="mb-3">Current Status</h5>
                <?php if ($hasFaceData): ?>
                    <div class="status-badge active">
                        ✓ Face ID Enabled
                    </div>
                    <p class="text-muted mt-2">Your face is registered for quick login.</p>
                    <button type="button" class="btn btn-danger w-100 mb-3" id="disable-face-btn">
                        <i data-feather="trash-2" class="me-2"></i> Remove Face Data
                    </button>
                <?php else: ?>
                    <div class="status-badge inactive">
                        ✕ Face ID Not Registered
                    </div>
                    <p class="text-muted mt-2">Register your face for quick and secure login.</p>
                <?php endif; ?>
                
                <!-- Info Box -->
                <div class="info-box">
                    <strong>💡 How it works:</strong>
                    <ul class="mb-0 mt-2" style="font-size: 0.95rem;">
                        <li>Register your face securely on your device</li>
                        <li>Use face recognition to login instantly</li>
                        <li>Your face data is encrypted and stored securely</li>
                        <li>Remove anytime from these settings</li>
                    </ul>
                </div>
                
                <!-- Register/Update Face -->
                <hr>
                <h5 class="mb-3">
                    <?php echo $hasFaceData ? 'Update' : 'Register'; ?> Face
                </h5>
                
                <div id="camera-section" style="display: none;">
                    <video id="face-video" autoplay playsinline></video>
                    <div id="face-status" class="text-center mb-3 p-2 rounded" style="background: rgba(0,122,255,0.1); color: #0056CC;">
                        Initializing camera...
                    </div>
                    <button type="button" class="btn btn-secondary w-100 mb-2" id="capture-face-btn">
                        Capture Face
                    </button>
                    <button type="button" class="btn btn-outline-secondary w-100" id="cancel-capture-btn">
                        Cancel
                    </button>
                </div>
                
                <button type="button" class="btn btn-success w-100" id="start-registration-btn">
                    <i data-feather="camera" class="me-2"></i>
                    <?php echo $hasFaceData ? 'Update Face' : 'Register Face'; ?>
                </button>
            </div>
        </div>
        
        <div class="mt-4 text-center">
            <a href="dashboard.php" class="text-muted">Back to Dashboard</a>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        feather.replace();
        
        let videoStream = null;
        
        document.getElementById('start-registration-btn').addEventListener('click', startRegistration);
        document.getElementById('cancel-capture-btn').addEventListener('click', cancelCapture);
        document.getElementById('capture-face-btn').addEventListener('click', captureFace);
        
        <?php if ($hasFaceData): ?>
        document.getElementById('disable-face-btn').addEventListener('click', disableFace);
        <?php endif; ?>
        
        function startRegistration() {
            const cameraSection = document.getElementById('camera-section');
            const video = document.getElementById('face-video');
            const status = document.getElementById('face-status');
            
            cameraSection.style.display = 'block';
            status.textContent = 'Accessing camera...';
            
            navigator.mediaDevices.getUserMedia({
                video: {
                    facingMode: 'user',
                    width: { ideal: 400 },
                    height: { ideal: 400 }
                }
            })
            .then(stream => {
                videoStream = stream;
                video.srcObject = stream;
                status.textContent = 'Camera ready - Position your face in the frame';
            })
            .catch(error => {
                status.textContent = 'Camera access denied';
                console.error('Camera error:', error);
            });
        }
        
        function captureFace() {
            const video = document.getElementById('face-video');
            const status = document.getElementById('face-status');
            
            status.textContent = 'Capturing face...';
            
            // Create a canvas and capture video frame
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            
            canvas.width = video.videoWidth;
            canvas.height = video.videoHeight;
            ctx.drawImage(video, 0, 0);
            
            // Create face descriptor from image data
            const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
            const faceDescriptor = btoa(JSON.stringify(Array.from(imageData.data.slice(0, 2000))));
            
            console.log('Face descriptor size:', faceDescriptor.length, 'bytes');
            
            status.textContent = 'Saving face...';
            
            fetch('../api/face_recognition.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=register_face&face_data=' + encodeURIComponent(faceDescriptor)
            })
            .then(r => {
                console.log('Response status:', r.status);
                return r.json();
            })
            .then(result => {
                console.log('API Response:', result);
                if (result.success) {
                    status.textContent = '✓ Face registered successfully!';
                    status.style.background = 'rgba(52, 199, 89, 0.2)';
                    status.style.color = '#00B741';
                    
                    setTimeout(() => {
                        cancelCapture();
                        location.reload();
                    }, 2000);
                } else {
                    status.textContent = 'Error: ' + (result.message || 'Unknown error');
                    status.style.background = 'rgba(255, 59, 48, 0.2)';
                    status.style.color = '#FF3B30';
                    console.error('API Error:', result.message);
                }
            })
            .catch(error => {
                status.textContent = 'Error: ' + error.message;
                console.error('Capture error:', error);
            });
        }
        
        function cancelCapture() {
            if (videoStream) {
                videoStream.getTracks().forEach(track => track.stop());
            }
            document.getElementById('camera-section').style.display = 'none';
            document.getElementById('face-status').textContent = '';
        }
        
        function disableFace() {
            if (confirm('Are you sure? You will need to use email/password to login.')) {
                fetch('../api/face_recognition.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'action=delete_face'
                })
                .then(r => r.json())
                .then(result => {
                    if (result.success) {
                        location.reload();
                    }
                });
            }
        }
    </script>
</body>
</html>
