<?php
// user/subscribe.php - User subscription purchase page with Razorpay integration
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';
require_once '../classes/SubscriptionManager.php';
require_once '../classes/RazorpayPayment.php';

$db = Database::getInstance();
$auth = new Auth($db);
$subscriptionMgr = new SubscriptionManager();

// Check if user is logged in
if (!$auth->isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

$currentUser = $auth->getCurrentUser();
$plans = $subscriptionMgr->getActivePlans();
$userSubscription = $subscriptionMgr->getActiveSubscription($currentUser['id']);
$trialInfo = $subscriptionMgr->getFreeTrial($currentUser['id']);

// Handle AJAX requests
if (isset($_GET['ajax']) && $_GET['ajax'] == '1') {
    if (isset($_GET['action']) && $_GET['action'] === 'get_plans') {
        // Return plans as JSON with colors for dashboard embedded display
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'success' => true,
            'plans' => array_map(function($plan) {
                return [
                    'id' => $plan['id'],
                    'plan_name' => $plan['plan_name'],
                    'duration_days' => $plan['duration_days'],
                    'search_limit' => $plan['search_limit'],
                    'price' => number_format($plan['price'], 0),
                    'card_color' => $plan['card_color'] ?? '#007AFF'
                ];
            }, $plans)
        ]);
        exit;
    }
}

// Handle payment verification (AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    if ($_POST['action'] === 'create_payment_order') {
        $planId = (int)($_POST['plan_id'] ?? 0);
        $plan = $db->fetch('SELECT * FROM subscription_plans WHERE id = ? AND is_active = true', [$planId]);
        
        if (!$plan) {
            echo json_encode(['success' => false, 'message' => 'Invalid plan']);
            exit;
        }
        
        // Create subscription record first - initially pending payment
        $subscriptionData = [
            'user_id' => $currentUser['id'],
            'plan_id' => $planId,
            'status' => 'expired',
            'payment_status' => 'pending',
            'start_date' => date('Y-m-d H:i:s'),
            'end_date' => date('Y-m-d H:i:s', strtotime('+' . $plan['duration_days'] . ' days')),
            'search_used' => 0
        ];
        
        error_log('Attempting to create subscription with data: ' . json_encode($subscriptionData));
        
        $subscriptionId = $db->insert('user_subscriptions', $subscriptionData);
        
        error_log('Subscription insert result: ' . ($subscriptionId ? 'SUCCESS (ID: ' . $subscriptionId . ')' : 'FAILED'));
        
        if (!$subscriptionId) {
            error_log('Failed to create subscription. Plan ID: ' . $planId . ', User ID: ' . $currentUser['id']);
            echo json_encode(['success' => false, 'message' => 'Failed to create subscription record']);
            exit;
        }
        
        // Create Razorpay order
        $razorpay = new RazorpayPayment(RAZORPAY_KEY_ID, RAZORPAY_KEY_SECRET);
        $orderResult = $razorpay->createOrder($currentUser['id'], $planId, $subscriptionId);
        
        if (!$orderResult['status']) {
            echo json_encode(['success' => false, 'message' => 'Failed to create payment order']);
            exit;
        }
        
        // Update subscription with Razorpay order ID
        $db->query(
            'UPDATE user_subscriptions SET razorpay_order_id = ? WHERE id = ?',
            [$orderResult['order_id'], $subscriptionId]
        );
        
        echo json_encode([
            'success' => true,
            'subscription_id' => $subscriptionId,
            'order_id' => $orderResult['order_id'],
            'amount' => $orderResult['amount'],
            'key_id' => $orderResult['key_id'],
            'plan_name' => $orderResult['plan_name']
        ]);
        exit;
    } 
    elseif ($_POST['action'] === 'verify_payment') {
        $orderId = $_POST['order_id'] ?? '';
        $paymentId = $_POST['payment_id'] ?? '';
        $signature = $_POST['signature'] ?? '';
        $subscriptionId = (int)($_POST['subscription_id'] ?? 0);
        
        if (!$orderId || !$paymentId || !$signature || !$subscriptionId) {
            echo json_encode(['success' => false, 'message' => 'Missing required parameters']);
            exit;
        }
        
        // Verify signature
        $razorpay = new RazorpayPayment(RAZORPAY_KEY_ID, RAZORPAY_KEY_SECRET);
        $verifyResult = $razorpay->verifyPaymentSignature($orderId, $paymentId, $signature);
        
        if (!$verifyResult['status'] || !$verifyResult['verified']) {
            // Mark subscription as failed
            $db->query(
                'UPDATE user_subscriptions SET payment_status = ?, status = ? WHERE id = ?',
                ['failed', 'expired', $subscriptionId]
            );
            
            echo json_encode(['success' => false, 'message' => 'Payment verification failed']);
            exit;
        }
        
        // Update subscription with payment details
        $updateData = [
            'razorpay_payment_id' => $paymentId,
            'razorpay_signature' => $signature,
            'payment_status' => 'success',
            'status' => 'active'
        ];
        
        $db->query(
            'UPDATE user_subscriptions SET razorpay_payment_id = ?, razorpay_signature = ?, payment_status = ?, status = ? WHERE id = ?',
            [$paymentId, $signature, 'success', 'active', $subscriptionId]
        );
        
        // If user had trial, mark it as ended
        if ($trialInfo && $trialInfo['is_active']) {
            $db->query('UPDATE free_trial_tracking SET is_trial_active = false, trial_ended_at = NOW() WHERE user_id = ?', 
                [$currentUser['id']]);
        }
        
        // Log subscription history
        $db->insert('subscription_history', [
            'user_id' => $currentUser['id'],
            'subscription_id' => $subscriptionId,
            'action' => 'payment_successful',
            'new_status' => 'active',
            'details' => json_encode([
                'razorpay_payment_id' => $paymentId,
                'plan_id' => $db->fetch('SELECT plan_id FROM user_subscriptions WHERE id = ?', [$subscriptionId])['plan_id']
            ])
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Payment successful! Your subscription is now active.',
            'redirect' => 'dashboard.php'
        ]);
        exit;
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Subscribe - Advanced Mobile Info System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #007AFF;
            --success: #34C759;
            --danger: #FF3B30;
            --warning: #FF9500;
            --light: #F2F2F7;
            --text-color: #1c1c1e;
            --border-color: #e5e5ea;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            background: var(--light);
            min-height: 100vh;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto;
            display: flex;
        }
        
        /* === Sidebar Styles === */
        .sidebar {
            width: 240px;
            background: white;
            border-right: 1px solid var(--border-color);
            padding-top: 0;
            box-shadow: 2px 0 8px rgba(0,0,0,0.05);
            position: fixed;
            left: 0;
            top: 0;
            height: 100vh;
            overflow-y: auto;
            z-index: 1000;
        }
        
        .sidebar::-webkit-scrollbar {
            width: 6px;
        }
        
        .sidebar::-webkit-scrollbar-track {
            background: transparent;
        }
        
        .sidebar::-webkit-scrollbar-thumb {
            background: #d0d0d0;
            border-radius: 3px;
        }
        
        .sidebar-header {
            padding: 1.25rem 1rem;
            border-bottom: 1px solid var(--border-color);
            text-align: center;
        }
        
        .sidebar-header .logo {
            font-weight: 700;
            color: var(--primary);
            font-size: 1.1rem;
        }
        
        .sidebar-menu {
            list-style: none;
            margin: 0;
            padding: 0;
        }
        
        .sidebar-menu a {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.9rem 1rem;
            color: var(--text-color);
            text-decoration: none;
            border-left: 3px solid transparent;
            transition: all 0.2s;
        }
        
        .sidebar-menu a:hover,
        .sidebar-menu a.active {
            background: rgba(0, 122, 255, 0.08);
            border-left-color: var(--primary);
            color: var(--primary);
        }
        
        .sidebar-menu i {
            width: 20px;
            height: 20px;
        }
        
        /* Main Content Area */
        .main-content {
            margin-left: 240px;
            flex: 1;
            padding: 20px;
            overflow-y: auto;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .header {
            text-align: center;
            color: var(--text-color);
            margin-bottom: 40px;
        }
        
        .header h1 {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 10px;
        }
        
        .header p {
            font-size: 1.1rem;
            opacity: 0.8;
        }
        
        .trial-status {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .trial-badge {
            display: inline-block;
            padding: 8px 16px;
            border-radius: 6px;
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .trial-badge.active {
            background: #D1FADF;
            color: #065F46;
        }
        
        .trial-badge.expired {
            background: #FEE2E2;
            color: #991B1B;
        }
        
        .plans-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 25px;
            margin-bottom: 40px;
        }
        
        .plan-card {
            background: white;
            border-radius: 16px;
            padding: 30px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .plan-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 8px 30px rgba(0,0,0,0.15);
        }
        
        .plan-card.active {
            border: 3px solid var(--primary);
            background: linear-gradient(135deg, rgba(0,122,255,0.05) 0%, transparent 100%);
        }
        
        .plan-card .badge {
            position: absolute;
            top: 15px;
            right: 15px;
            background: var(--primary);
            color: white;
            padding: 6px 12px;
            border-radius: 6px;
            font-size: 0.75rem;
        }
        
        .plan-header {
            margin-bottom: 20px;
        }
        
        .plan-name {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1F2937;
            margin-bottom: 5px;
        }
        
        .plan-duration {
            color: #6B7280;
            font-size: 0.95rem;
        }
        
        .plan-price {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--primary);
            margin: 20px 0;
        }
        
        .plan-price .currency {
            font-size: 1.5rem;
            vertical-align: super;
        }
        
        .plan-features {
            list-style: none;
            padding: 0;
            margin: 20px 0;
        }
        
        .plan-features li {
            padding: 10px 0;
            color: #4B5563;
            border-bottom: 1px solid #E5E7EB;
            font-size: 0.95rem;
        }
        
        .plan-features li:last-child {
            border-bottom: none;
        }
        
        .plan-features i {
            color: var(--success);
            margin-right: 10px;
            width: 20px;
        }
        
        .subscribe-btn {
            width: 100%;
            padding: 12px;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            font-size: 1rem;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 20px;
            pointer-events: auto;
        }
        
        .subscribe-btn.btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .subscribe-btn.btn-primary:hover {
            background: #0051D5;
            transform: scale(1.02);
        }
        
        .subscribe-btn.btn-secondary {
            background: #E5E7EB;
            color: #1F2937;
        }
        
        .subscribe-btn.btn-secondary:hover {
            background: #D1D5DB;
        }
        
        .info-section {
            background: white;
            border-radius: 12px;
            padding: 25px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .info-section h3 {
            font-size: 1.3rem;
            font-weight: 700;
            margin-bottom: 15px;
            color: #1F2937;
        }
        
        .info-section ul {
            list-style: none;
            padding: 0;
        }
        
        .info-section li {
            padding: 10px 0;
            color: #4B5563;
        }
        
        .info-section li:before {
            content: "✓ ";
            color: var(--success);
            font-weight: 700;
            margin-right: 10px;
        }
        
        .loading {
            display: none;
        }
        
        .loading.show {
            display: block;
        }
    </style>
</head>
<body>
    <!-- Sidebar Navigation -->
    <nav class="sidebar">
        <div class="sidebar-header">
            <div class="logo">Simple Checks</div>
        </div>
        <ul class="sidebar-menu">
            <li><a href="../user/dashboard.php"><i class="fas fa-home"></i> Dashboard</a></li>
            <li><a href="../user/dashboard.php#mobile-search"><i class="fas fa-mobile-alt"></i> Mobile Search</a></li>
            <li><a href="../user/dashboard.php#aadhaar-search"><i class="fas fa-id-card"></i> Aadhaar Search</a></li>
            <li><a href="../user/dashboard.php"><i class="fas fa-crown"></i> Premium</a></li>
            <li><a href="../user/payments.php"><i class="fas fa-money-bill-wave"></i> My Transactions</a></li>
            <li><a href="../user/profile.php"><i class="fas fa-user"></i> Profile</a></li>
            <li><a href="../user/support.php"><i class="fas fa-headset"></i> Support</a></li>
            <li><a href="../logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
        </ul>
    </nav>

    <!-- Main Content -->
    <div class="main-content">
        <div class="container">
            <div class="header">
                <h1><i class="fas fa-crown"></i> Premium Subscription</h1>
            <p>Unlock unlimited searches with our flexible plans</p>
        </div>
        
        <!-- Trial Status -->
        <?php if ($trialInfo): ?>
            <div class="trial-status">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <strong>Free Trial Status:</strong>
                        <span class="trial-badge <?php echo $trialInfo['is_active'] ? 'active' : 'expired'; ?>">
                            <?php 
                            if ($trialInfo['is_active']) {
                                echo '✓ Active - ' . $trialInfo['searches_remaining'] . ' searches remaining';
                            } else {
                                echo '✗ Expired - Subscribe to continue';
                            }
                            ?>
                        </span>
                    </div>
                    <div>
                        <strong>Started:</strong> <?php echo date('d M Y', strtotime($trialInfo['started_at'])); ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <!-- Current Subscription -->
        <?php if ($userSubscription && $userSubscription['status'] === 'active'): ?>
            <div class="trial-status" style="background: #D1FADF;">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <strong style="color: #065F46;">Active Subscription:</strong>
                        <span style="color: #065F46; font-weight: 600;">
                            <?php echo htmlspecialchars($userSubscription['plan_name'] ?? 'Premium Plan'); ?>
                        </span>
                    </div>
                    <div style="color: #065F46;">
                        <strong>Expires:</strong> <?php echo date('d M Y', strtotime($userSubscription['end_date'])); ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <!-- Plans Grid -->
        <div class="plans-grid">
            <?php foreach ($plans as $plan): ?>
                <div class="plan-card <?php echo ($userSubscription && $userSubscription['plan_id'] === $plan['id']) ? 'active' : ''; ?>">
                    <?php if ($userSubscription && $userSubscription['plan_id'] === $plan['id']): ?>
                        <div class="badge">Current Plan</div>
                    <?php endif; ?>
                    
                    <div class="plan-header">
                        <div class="plan-name"><?php echo htmlspecialchars($plan['plan_name']); ?></div>
                        <div class="plan-duration">
                            <i class="far fa-calendar"></i> <?php echo $plan['duration_days']; ?> Days
                        </div>
                    </div>
                    
                    <div class="plan-price">
                        <span class="currency">₹</span><?php echo number_format($plan['price'], 0); ?>
                    </div>
                    
                    <ul class="plan-features">
                        <li>
                            <i class="fas fa-check"></i>
                            <?php echo $plan['search_limit']; ?> Searches
                        </li>
                        <li>
                            <i class="fas fa-check"></i>
                            <?php echo $plan['duration_days']; ?> Days Access
                        </li>
                        <li>
                            <i class="fas fa-check"></i>
                            Aadhar Search Support
                        </li>
                        <li>
                            <i class="fas fa-check"></i>
                            Mobile Search Support
                        </li>
                        <li>
                            <i class="fas fa-check"></i>
                            Priority Support
                        </li>
                    </ul>
                    
                    <?php if ($plan['description']): ?>
                        <p style="color: #6B7280; font-size: 0.9rem; font-style: italic; margin: 15px 0;">
                            <?php echo htmlspecialchars($plan['description']); ?>
                        </p>
                    <?php endif; ?>
                    
                    <button type="button" class="subscribe-btn btn-primary" onclick="subscribeToPlan(<?php echo $plan['id']; ?>, <?php echo number_format($plan['price'], 2); ?>); return false;">
                        Subscribe Now
                    </button>
                </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Info Section -->
        <div class="info-section">
            <h3><i class="fas fa-shield-alt"></i> Why Choose Our Premium Plans?</h3>
            <ul style="columns: 2;">
                <li>Secure and encrypted data transmission</li>
                <li>24/7 customer support</li>
                <li>Instant result delivery</li>
                <li>No hidden charges</li>
                <li>Flexible renewal options</li>
                <li>Money-back guarantee</li>
            </ul>
        </div>
    </div>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://checkout.razorpay.com/v1/checkout.js"></script>
    
    <script>
        // Make sure jQuery is loaded
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Subscribe page loaded');
        });
        
        function subscribeToPlan(planId, amount) {
            console.log('Subscribe function called with planId:', planId, 'amount:', amount);
            
            // Step 1: Create payment order
            $.ajax({
                type: 'POST',
                url: 'subscribe.php',
                data: {
                    action: 'create_payment_order',
                    plan_id: planId
                },
                dataType: 'json',
                success: function(response) {
                    console.log('Order creation response:', response);
                    if (response.success) {
                        openPaymentWindow(response);
                    } else {
                        alert('Error: ' + response.message);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', status, error);
                    alert('Error creating payment order: ' + error);
                }
            });
            return false;
        }
        
        function openPaymentWindow(orderData) {
            console.log('Opening payment window with:', orderData);
            
            if (typeof Razorpay === 'undefined') {
                alert('Payment gateway not loaded. Please refresh the page.');
                return;
            }
            
            var options = {
                key: orderData.key_id,
                amount: orderData.amount * 100,
                currency: 'INR',
                name: 'Advanced Mobile Info System',
                description: orderData.plan_name,
                order_id: orderData.order_id,
                handler: function(response) {
                    console.log('Payment response:', response);
                    verifyPayment(response, orderData.subscription_id);
                },
                prefill: {
                    email: '<?php echo htmlspecialchars($currentUser['email']); ?>',
                    contact: '<?php echo htmlspecialchars($currentUser['phone'] ?? ''); ?>'
                },
                theme: {
                    color: '#007AFF'
                }
            };
            
            var rzp = new Razorpay(options);
            rzp.open();
        }
        
        function verifyPayment(paymentResponse, subscriptionId) {
            $.ajax({
                type: 'POST',
                url: 'subscribe.php',
                data: {
                    action: 'verify_payment',
                    subscription_id: subscriptionId,
                    order_id: paymentResponse.razorpay_order_id,
                    payment_id: paymentResponse.razorpay_payment_id,
                    signature: paymentResponse.razorpay_signature
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        alert('✓ ' + response.message);
                        setTimeout(function() {
                            window.location.href = response.redirect;
                        }, 1500);
                    } else {
                        alert('Payment verification failed: ' + response.message);
                    }
                },
                error: function() {
                    alert('Error verifying payment');
                }
            });
        }
    </script>
        </div><!-- end container -->
    </div><!-- end main-content -->
</body>
</html>
