<?php
session_start();
require_once 'config/config.php';
require_once 'classes/Database.php';
require_once 'classes/Auth.php';
require_once 'classes/SubscriptionManager.php';
require_once 'classes/RazorpayPayment.php';

$db = Database::getInstance();
$auth = new Auth($db);

// Check if user is logged in
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit();
}

$userId = $_SESSION['user_id'];
$subscriptionMgr = new SubscriptionManager();

// Get Razorpay credentials from config
$razorpay = new RazorpayPayment(RAZORPAY_KEY_ID, RAZORPAY_KEY_SECRET);

// Get user's trial and subscription
$freeTrial = $subscriptionMgr->getFreeTrial($userId);
$activeSubscription = $subscriptionMgr->getActiveSubscription($userId);
$userSubscriptions = $subscriptionMgr->getUserSubscriptions($userId);
$plans = $subscriptionMgr->getActivePlans();

$message = '';
$message_type = '';

// Handle plan purchase
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'purchase_plan') {
    $planId = (int)$_POST['plan_id'];
    
    // Create subscription
    $subscriptionResult = $subscriptionMgr->createSubscription($userId, $planId, 'pending_order_id');
    
    if ($subscriptionResult['status']) {
        $subscriptionId = $subscriptionResult['subscription_id'];
        
        // Create Razorpay order
        $orderResult = $razorpay->createOrder($userId, $planId, $subscriptionId);
        
        if ($orderResult['status']) {
            // Store order details in session for Razorpay checkout
            $_SESSION['razorpay_order'] = [
                'order_id' => $orderResult['order_id'],
                'subscription_id' => $subscriptionId,
                'plan_id' => $planId,
                'amount' => $orderResult['amount'],
                'key_id' => $orderResult['key_id']
            ];
            
            // Redirect to payment page
            header('Location: process_payment.php?order_id=' . $orderResult['order_id']);
            exit();
        } else {
            $message = 'Failed to create payment order';
            $message_type = 'danger';
        }
    } else {
        $message = 'Failed to create subscription';
        $message_type = 'danger';
    }
}

// Handle subscription cancellation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'cancel_subscription') {
    $subscriptionId = (int)$_POST['subscription_id'];
    
    $result = $subscriptionMgr->cancelSubscription($userId, $subscriptionId);
    if ($result['status']) {
        $message = 'Subscription cancelled successfully';
        $message_type = 'success';
        // Refresh data
        $activeSubscription = $subscriptionMgr->getActiveSubscription($userId);
        $userSubscriptions = $subscriptionMgr->getUserSubscriptions($userId);
    } else {
        $message = 'Failed to cancel subscription';
        $message_type = 'danger';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Subscriptions</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        :root {
            --primary: #007AFF;
            --success: #34C759;
            --danger: #FF3B30;
            --warning: #FF9500;
        }
        
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto;
        }
        
        .container-custom {
            max-width: 1200px;
            margin-top: 2rem;
            margin-bottom: 2rem;
        }
        
        .header {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        
        .card {
            border: none;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            transition: all 0.3s;
            margin-bottom: 1.5rem;
        }
        
        .card:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.12);
            transform: translateY(-2px);
        }
        
        .subscription-card {
            border-left: 4px solid var(--primary);
            padding: 1.5rem;
        }
        
        .subscription-card.active {
            border-left-color: var(--success);
            background: linear-gradient(135deg, rgba(52,199,89,0.05) 0%, rgba(52,199,89,0.02) 100%);
        }
        
        .badge {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            font-size: 0.875rem;
            font-weight: 600;
        }
        
        .badge-active {
            background: #d4edda;
            color: #155724;
        }
        
        .badge-expired {
            background: #f8d7da;
            color: #721c24;
        }
        
        .progress-bar {
            background: var(--primary);
        }
        
        .plan-card {
            border: 2px solid #e5e5ea;
            border-radius: 12px;
            padding: 1.5rem;
            transition: all 0.3s;
            text-align: center;
        }
        
        .plan-card:hover {
            border-color: var(--primary);
            box-shadow: 0 4px 12px rgba(0, 122, 255, 0.2);
        }
        
        .plan-card.recommended {
            border-color: var(--primary);
            background: linear-gradient(135deg, rgba(0, 122, 255, 0.05) 0%, rgba(0, 122, 255, 0.02) 100%);
        }
        
        .plan-price {
            font-size: 2rem;
            font-weight: 700;
            color: var(--primary);
            margin: 1rem 0;
        }
        
        .trial-info {
            background: linear-gradient(135deg, #fff9e6 0%, #fff3cc 100%);
            border-left: 4px solid var(--warning);
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
        }
        
        .alert {
            border-radius: 12px;
            border: none;
        }
        
        .btn-primary {
            background: var(--primary);
            border: none;
            border-radius: 8px;
            padding: 0.75rem 1.5rem;
            font-weight: 600;
        }
        
        .btn-primary:hover {
            background: #0051cc;
        }
        
        .btn-outline-primary {
            color: var(--primary);
            border: 2px solid var(--primary);
        }
        
        .btn-outline-primary:hover {
            background: var(--primary);
            color: white;
        }
    </style>
</head>
<body>
    <div class="container container-custom">
        <!-- Header -->
        <div class="header d-flex justify-content-between align-items-center">
            <div>
                <h1 class="mb-0">💳 My Subscriptions</h1>
                <p class="text-muted mb-0">Manage your subscription plans</p>
            </div>
            <a href="dashboard.php" class="btn btn-outline-primary">
                <i data-feather="arrow-left" style="width: 18px; height: 18px;"></i> Back
            </a>
        </div>
        
        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
                <?php echo htmlspecialchars($message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>
        
        <!-- Free Trial Info -->
        <?php if ($freeTrial && $freeTrial['is_active']): ?>
            <div class="trial-info">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <h5 class="mb-2">🎉 Free Trial Active</h5>
                        <p class="mb-2">You have <strong><?php echo $freeTrial['searches_remaining']; ?> searches remaining</strong> from your free trial.</p>
                        <small class="text-muted">Started: <?php echo date('M d, Y', strtotime($freeTrial['started_at'])); ?></small>
                    </div>
                    <div>
                        <div class="progress" style="width: 200px;">
                            <div class="progress-bar" role="progressbar" style="width: <?php echo (($freeTrial['searches_used'] / $freeTrial['searches_limit']) * 100); ?>%" aria-valuenow="<?php echo $freeTrial['searches_used']; ?>" aria-valuemin="0" aria-valuemax="<?php echo $freeTrial['searches_limit']; ?>"></div>
                        </div>
                        <small class="text-muted d-block mt-2"><?php echo $freeTrial['searches_used']; ?> / <?php echo $freeTrial['searches_limit']; ?> used</small>
                    </div>
                </div>
            </div>
        <?php elseif ($freeTrial): ?>
            <div class="alert alert-warning">
                ℹ️ Your free trial has ended. Subscribe to a plan to continue using our services.
            </div>
        <?php endif; ?>
        
        <!-- Active Subscription -->
        <?php if ($activeSubscription): ?>
            <div class="card subscription-card active">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <h5 class="mb-2">
                            <i data-feather="check-circle" style="width: 20px; height: 20px; color: var(--success); margin-right: 0.5rem;"></i>
                            <?php echo htmlspecialchars($activeSubscription['plan_name']); ?>
                        </h5>
                        <p class="text-muted mb-2">
                            Valid from <?php echo date('M d', strtotime($activeSubscription['start_date'])); ?> to <?php echo date('M d, Y', strtotime($activeSubscription['end_date'])); ?>
                        </p>
                        <div class="progress mb-2" style="height: 8px;">
                            <div class="progress-bar" role="progressbar" style="width: <?php echo (($activeSubscription['searches_used'] / $activeSubscription['searches_limit']) * 100); ?>%" aria-valuenow="<?php echo $activeSubscription['searches_used']; ?>" aria-valuemin="0" aria-valuemax="<?php echo $activeSubscription['searches_limit']; ?>"></div>
                        </div>
                        <small class="text-muted">
                            <?php echo $activeSubscription['searches_used']; ?> / <?php echo $activeSubscription['searches_limit']; ?> searches used
                        </small>
                    </div>
                    <div class="text-end">
                        <span class="badge badge-active">Active</span>
                        <p class="mt-2 mb-0">
                            <strong><?php echo $activeSubscription['days_remaining']; ?> days remaining</strong>
                        </p>
                        <form method="POST" style="display: inline; margin-top: 1rem;">
                            <input type="hidden" name="action" value="cancel_subscription">
                            <input type="hidden" name="subscription_id" value="<?php echo $activeSubscription['id']; ?>">
                            <button type="submit" class="btn btn-sm btn-outline-danger" onclick="return confirm('Cancel subscription?');">
                                Cancel
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <!-- Available Plans -->
        <h3 class="mb-3">Available Plans</h3>
        <div class="row">
            <?php foreach ($plans as $plan): ?>
                <div class="col-md-6 col-lg-4">
                    <div class="plan-card <?php echo ($activeSubscription && $activeSubscription['plan_id'] == $plan['id']) ? 'current' : ''; ?>">
                        <h5><?php echo htmlspecialchars($plan['plan_name']); ?></h5>
                        <p class="text-muted mb-2"><?php echo htmlspecialchars($plan['description']); ?></p>
                        
                        <div class="plan-price">₹<?php echo number_format($plan['price'], 0); ?></div>
                        
                        <ul class="list-unstyled mb-3 text-start">
                            <li class="mb-2">
                                <i data-feather="check" style="width: 18px; height: 18px; color: var(--success); margin-right: 0.5rem;"></i>
                                <strong><?php echo $plan['search_limit']; ?> searches</strong>
                            </li>
                            <li class="mb-2">
                                <i data-feather="check" style="width: 18px; height: 18px; color: var(--success); margin-right: 0.5rem;"></i>
                                <strong><?php echo $plan['duration_days']; ?> days</strong> validity
                            </li>
                            <li>
                                <i data-feather="check" style="width: 18px; height: 18px; color: var(--success); margin-right: 0.5rem;"></i>
                                Priority support
                            </li>
                        </ul>
                        
                        <?php if ($activeSubscription && $activeSubscription['plan_id'] == $plan['id']): ?>
                            <button class="btn btn-primary w-100" disabled>
                                ✓ Current Plan
                            </button>
                        <?php else: ?>
                            <form method="POST">
                                <input type="hidden" name="action" value="purchase_plan">
                                <input type="hidden" name="plan_id" value="<?php echo $plan['id']; ?>">
                                <button type="submit" class="btn btn-primary w-100">
                                    Subscribe Now
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Subscription History -->
        <?php if ($userSubscriptions): ?>
            <div class="mt-4">
                <h3 class="mb-3">Subscription History</h3>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead class="table-light">
                            <tr>
                                <th>Plan</th>
                                <th>Price</th>
                                <th>Duration</th>
                                <th>Status</th>
                                <th>Valid Till</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($userSubscriptions as $sub): ?>
                                <tr>
                                    <td><strong><?php echo htmlspecialchars($sub['plan_name']); ?></strong></td>
                                    <td>₹<?php echo number_format($sub['price'], 2); ?></td>
                                    <td><?php echo $sub['duration_days']; ?> days</td>
                                    <td>
                                        <span class="badge <?php echo ($sub['status'] === 'active' && $sub['end_date'] > date('Y-m-d H:i:s')) ? 'badge-active' : 'badge-expired'; ?>">
                                            <?php echo ucfirst($sub['status']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('M d, Y', strtotime($sub['end_date'])); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        feather.replace();
    </script>
</body>
</html>
